'use strict';

var assert = require('chai').assert;
var expect = require('chai').expect;

var proxyquire = require('proxyquire').noCallThru().noPreserveCache();
var Status = require('../../../../../mocks/Status');
var Session = require('../../../../../mocks/Session');
var Order = require('../../../../../mocks/dw/order/Order');
var Resource = require('../../../../../mocks/dw/web/Resource');
var ProductMgr = require('../../../../../mocks/dw/catalog/ProductMgr');
var Site = require('../../../../../mocks/dw/system/Site');
var ShippingMgr = require('../../../../../mocks/dw/order/ShippingMgr');
var OrderMgr = require('../../../../../mocks/dw/order/OrderMgr');
var System = require('../../../../../mocks/dw/system/System');
var StringUtils = require('../../../../../mocks/dw/util/StringUtils');
var OauthFactory = require('../../../../../mocks/util/OauthFactory');
var OauthService = require('../../../../../mocks/services/OauthService');
var OrderUtilCode = require('../../../../../mocks/util/OrderUtilCode');
var FixedPriceShippingDiscount = require('../../../../../mocks/dw/campaign/FixedPriceShippingDiscount');

var Basket = require('../../../../../mocks/Basket');

describe('OrderCreateHelper', function () {
    global.session = new Session.forms();

    var OrderCreateHelper = proxyquire("../../../../../../social_checkout/cartridges/int_social_checkout/cartridge/scripts/util/OrderCreateHelper.js", {
        'dw/value/Money': function () {},
        'dw/order/Order': Order,
        'dw/web/Resource': Resource,
        'dw/catalog/ProductMgr': ProductMgr,
        'dw/system/Site': Site,
        'dw/util/StringUtils': StringUtils,
        'dw/system/System': System,
        '~/cartridge/scripts/util/OauthFactory': OauthFactory,
        '~/cartridge/scripts/services/OauthService': OauthService,
        '~/cartridge/scripts/util/OrderUtilCode': OrderUtilCode,
        'dw/order/ShippingMgr': ShippingMgr,
        'dw/order/OrderMgr': OrderMgr,
        'dw/campaign/FixedPriceShippingDiscount': FixedPriceShippingDiscount
    });

    it('convertState: Convert state code to state name', function() {
        var stateName = OrderCreateHelper.convertState('AL', 'name');
        assert.equal(stateName, 'Alabama');
    });

    it('convertState: Convert state name to state code', function() {
        var stateCode = OrderCreateHelper.convertState('Alabama', 'code');
        assert.equal(stateCode, 'AL');
    });

    it('convertState: Try to convert state code to state name but recieve a null code', function() {
        var stateName = OrderCreateHelper.convertState(null, 'name');
        assert.equal(stateName, '');
    });

    it('convertCountry: Convert country code to country name', function() {
        var countryName = OrderCreateHelper.convertCountry('VU', 'name');
        assert.equal(countryName, 'Vanuatu');
    });

    it('convertCountry: Try to convert country code to country name but recieve a null code', function() {
        var stateName = OrderCreateHelper.convertCountry(null, 'name');
        assert.equal(stateName, '');
    });

    it('setBillingAddress: set the billing address for the order', function() {
        var basket = new Basket();

        var shippingAddressJson = {
            address1: '344 Clinton Street,',
            address2:  'Apartment 3D',
            city: 'Metropolis',
            postal_code: '04330',
            state_code: 'NY',
            country_code: {
                displayValue: 'United States',
                value: 'US'
            },
            phone: '333-333-3333'
        };
        
        var customerObject = {
            firstName: 'Clark',
            lastName: 'Kent'
        }
        
        var billingAddressJson = null;


        var testFn = OrderCreateHelper.setBillingAddress;
        expect(function(){testFn(basket, billingAddressJson, shippingAddressJson, customerObject)}).to.not.throw();
    });

    it('setBillingAddress: set the billing address for the order with only a first name', function() {
        var basket = new Basket();

        var shippingAddressJson = {
            address1: '344 Clinton Street,',
            address2:  'Apartment 3D',
            city: 'Metropolis',
            postal_code: '04330',
            state_code: 'NY',
            country_code: {
                displayValue: 'United States',
                value: 'US'
            },
            phone: '333-333-3333'
        };
        
        var customerObject = {
            firstName: 'Clark',
            lastName: 'Kent'
        }
        
        var billingAddressJson = {
            first_name: 'Clark Kent',
            last_name: '',
            address1: '344 Clinton Street,',
            address2:  'Apartment 3D',
            city: 'Metropolis',
            postal_code: '04330',
            state_code: 'NY',
            country_code: {
                displayValue: 'United States',
                value: 'US'
            },
            phone: '333-333-3333'
        };

        var testFn = OrderCreateHelper.setBillingAddress;
        expect(function(){testFn(basket, billingAddressJson, shippingAddressJson, customerObject)}).to.not.throw();
    });
    
    it('setShippingAddress: set the shipping address for the order', function() {
        var basket = new Basket();
        var shipmentJson = {
            shipping_address: {
                first_name: 'Clark Kent',
                last_name: '',
                address1: '344 Clinton Street,',
                address2:  'Apartment 3D',
                city: 'Metropolis',
                postal_code: '04330',
                state_code: 'NY',
                country_code: {
                    displayValue: 'United States',
                    value: 'US'
                },
                phone: '333-333-3333'
            }
        }

        var testFn = OrderCreateHelper.setShippingAddress;
        expect(function(){testFn(basket, shipmentJson)}).to.not.throw();
    });

    it('setShippingAddress: set the shipping address without shipping_address', function() {
        var basket = new Basket();
        var shipmentJson = {
            shipping_address: null
        }
        
        var result = OrderCreateHelper.setShippingAddress(basket, shipmentJson);
        
        assert.equal(result.hadCreateShippingAddress, true);
    });

    it('getCustomerName: setting from billing address', function() {
        var basket = new Basket();

        var customerObject = OrderCreateHelper.getCustomerName(basket, null, null);
        assert.equal(customerObject.firstName, basket.billingAddress.firstName);
        assert.equal(customerObject.lastName, basket.billingAddress.lastName);
    });

    it('getCustomerName: setting from order customer profile', function() {
        var basket = new Basket();
        var orderCustomer = {
            profile: {
                firstName: 'Vandal',
                lastName: 'Savage'
            }
        };

        var customerObject = OrderCreateHelper.getCustomerName(null, null, orderCustomer);
        assert.equal(customerObject.firstName, orderCustomer.profile.firstName);
        assert.equal(customerObject.lastName, orderCustomer.profile.lastName);
    });

    it('getCustomerName: setting from Json data billing address', function() {
        var basket = new Basket();
        var orderJSON = {
            customer: {
                billing_address: {
                    first_name: "Bruce",
                    last_name: "Wayne",
                }
            }
        }

        var customerObject = OrderCreateHelper.getCustomerName(null, orderJSON, null);
        assert.equal(customerObject.firstName, orderJSON.customer.billing_address.first_name);
        assert.equal(customerObject.lastName, orderJSON.customer.billing_address.last_name);
    });

    it('getCustomerName: setting from Json data customer info', function() {
        var basket = new Basket();
        var orderJSON = {
            customer: {
                customer_name: "Hank McCoy",
            }
        }
        

        var customerObject = OrderCreateHelper.getCustomerName(null, orderJSON, null);
        assert.equal(customerObject.firstName, 'Hank');
        assert.equal(customerObject.lastName, 'McCoy');
    });

    it('getCustomerName: setting from Json data customer info with just 1 word first name', function() {
        var basket = new Basket();
        var orderJSON = {
            customer: {
                customer_name: 'Lara',
            }
        }
        
        var customerObject = OrderCreateHelper.getCustomerName(null, orderJSON, null);
        assert.equal(customerObject.firstName, 'Lara');
        assert.equal(customerObject.lastName, 'Shop'); // based on var DEFAULT_VALUES defined on OrderCreateHelper.js scripts
    });

    it('getCustomerName: setting from Json data shipping address', function() {
        var basket = new Basket();

        var orderJSON = {
            customer: {
            },
            shipment: {
                shipping_method: "001",
                shipping_address: {
                    first_name: "Return",
                    last_name: "Johnson",
                    address1: "31 Ocean Ave",
                    city: "Santa Monoca",
                    postal_code: "90211",
                    state_code: "CA",
                    country_code: "US",
                    phone: "6718891114"
                }
            }                
        }
        
        var customerObject = OrderCreateHelper.getCustomerName(null, orderJSON, null);

        assert.equal(customerObject.firstName, orderJSON.shipment.shipping_address.first_name);
        assert.equal(customerObject.lastName, orderJSON.shipment.shipping_address.last_name);
    });

    it('getCustomerName: fallback to defaults', function() {
        var basket = new Basket();
        
        var customerObject = OrderCreateHelper.getCustomerName(null, null, null);

        assert.equal(customerObject.firstName, 'TikTok');
        assert.equal(customerObject.lastName, 'Shop');
    });
    
    it('addItemsToBasket: Error adding item', function() {
        var itensToBasket = OrderCreateHelper.addItemsToBasket(null);
        assert.equal(itensToBasket.error, true);
        assert.equal(itensToBasket.msg, "Error adding item : TypeError: Cannot read properties of null (reading 'getCurrencyCode')");
    });

    it('addItemsToBasket: Error "Item not available for the given social channel"', function() {
        var basket = new Basket();

        var shipment = {};

        var productItems = [
            {
                product_id: "noproduct",
                quantity: "3",
                net_price: "1000",
                tax_rate: "10",
                tax: "10",
                price_adjustments: "500"
            }
        ]

        var defaultPromoID = "somePromoId";

        var itensToBasket = OrderCreateHelper.addItemsToBasket(basket, shipment, productItems, null, defaultPromoID);

        assert.equal(itensToBasket.error, true);
        assert.equal(itensToBasket.msg, 'Item not available for the given social channel');
    });

    it('addItemsToBasket: Error "Insufficient qty available"', function() {
        var basket = new Basket();

        var shipment = {};
        var productItems = [
            {
                product_id: "000001",
                quantity: "3",
                net_price: "1000",
                tax_rate: "10",
                tax: "10",
                price_adjustments: "500"
            }
        ]
        var sInventory = {
            getRecord: function() {
                return {
                    ATS: {
                        value: 0
                    }
                }
            }
        };

        var defaultPromoID = "somePromoId";

        var itensToBasket = OrderCreateHelper.addItemsToBasket(basket, shipment, productItems, sInventory, defaultPromoID);

        assert.equal(itensToBasket.error, true);
        assert.equal(itensToBasket.msg, 'Insufficient qty available');
    });

    it('addItemsToBasket: Item added to cart with productItems.tax_rate 0', function() {
        var basket = new Basket();
        var shipment = {};
        var productItems = [
            {
                product_id: "000001",
                quantity: "3",
                net_price: "1000",
                tax_rate: "0",
                tax: "10",
                price_adjustments: [
                    {
                        net_price: 2000,
                        promotion_id: "thisPromoId",
                        tax: 10
                        
                    }
                ]
            }
        ]
        var sInventory = {
            getRecord: function() {
                return {
                    ATS: {
                        value: 1000
                    }
                }
            }
        };

        var defaultPromoID = "somePromoId";

        var itensToBasket = OrderCreateHelper.addItemsToBasket(basket, shipment, productItems, sInventory, defaultPromoID);

        assert.equal(itensToBasket.msg, 'item added to cart');
    });

    it('addItemsToBasket: Item added to cart', function() {
        var basket = new Basket();
        var shipment = {};
        var productItems = [
            {
                product_id: "000001",
                quantity: "3",
                net_price: "1000",
                tax_rate: "10",
                tax: "10",
                price_adjustments: [
                    {
                        net_price: 2000,
                        promotion_id: "thisPromoId",
                        tax: 10
                        
                    }
                ]
            }
        ]
        var sInventory = {
            getRecord: function() {
                return {
                    ATS: {
                        value: 1000
                    }
                }
            }
        };

        var defaultPromoID = "somePromoId";

        var itensToBasket = OrderCreateHelper.addItemsToBasket(basket, shipment, productItems, sInventory, defaultPromoID);

        assert.equal(itensToBasket.error, false);
        assert.equal(itensToBasket.msg, 'item added to cart');
    });

    it('setShippingMethod: set the shipping method', function() {
        
        var shipment = {
            setShippingMethod: function() {
                return;
            }
        };

        var shipmentJson = {
            shipping_method: '0001'
        };

        var testFn = OrderCreateHelper.setShippingMethod;
        expect(function(){testFn(shipment, shipmentJson)}).to.not.throw();
    });

    it('setShippingCost: set shipping cost', function() {

        var settings = {
            priceValue: 0,
            tax: 0,
            setPriceValue: function() {
                this.priceValue = 1000;
            },
            updateTax: function() {
                this.tax = 1.0;
            }
        }
        
        var shipment = {
            getStandardShippingLineItem: function() {
                return settings;
            }
        };

        var shipmentJson = {
            net_price: '123'
        };


        OrderCreateHelper.setShippingCost(shipment, shipmentJson);

        assert.equal(shipment.getStandardShippingLineItem().priceValue, 1000);
        assert.equal(shipment.getStandardShippingLineItem().tax, 1.0);
    });

    it('removePriceAdjustments: remove any auto applied promotion/price adjustments', function() {
        
        var basket = new Basket();
        var defaultPromoID = '';

        var testFn = OrderCreateHelper.removePriceAdjustments;
        expect(function(){testFn(basket, defaultPromoID)}).to.not.throw();
    });

    it('checkAccessToken: invalid access token 1', function() {

        var req = {
            httpHeaders: {
                containsKey: function() {
                    return false;
                }
            }
        }

        var response = OrderCreateHelper.checkAccessToken(req);

        assert.equal(response.status, 1);
        assert.equal(response.type, 'InvalidAccessTokenException');
        assert.equal(response.msg, 'Unauthorized request! The access token is invalid.');
    });

    it('checkAccessToken: invalid access token 2', function() {

        var req = {
            httpHeaders: {
                containsKey: function() {
                    return true;
                },
                get: function() {
                    return {
                        substring: function() {
                            return 'error';
                        }    
                    }
                }
            }
        }

        var response = OrderCreateHelper.checkAccessToken(req);

        assert.equal(response.status, 1);
        assert.equal(response.type, 'InvalidAccessTokenException');
        assert.equal(response.msg, 'Unauthorized request! The access token is invalid.');
    });

    it('checkAccessToken: check access token', function() {

        var req = {
            httpHeaders: {
                containsKey: function() {
                    return true;
                },
                get: function() {
                    return 'accessToken';
                }
            }
        }

        var response = OrderCreateHelper.checkAccessToken(req);

        assert.equal(response.status, 0);
        assert.equal(response.code, 'SUCCESS');
        assert.equal(response.msg, 'Shipment(s) successfully updated.');
    });

    it('getSitePreference: insert a site preference key and recieve SitePreference value', function() {
        global.empty = function(str) {
            return (!str || str === '') ? true : false;
        }
            
        var key = 'SOME_API_KEY'

        var response = OrderCreateHelper.getSitePreference(key);

        assert.equal(response, 'SOME_API_KEY');
    });

    it('getSitePreference: insert null as key and recieve no value', function() {
        global.empty = function(str) {
            return (!str || str === '') ? true : false;
        }
            
        var key = null;

        var response = OrderCreateHelper.getSitePreference(key);

        assert.equal(response, '');
    });

    it('orderExists: insert external order ID and recieve a valid response', function() {
        var extOrdID = '000001';
        var response = OrderCreateHelper.orderExists(extOrdID);

        assert.equal(response.orderNo, '000001');
        assert.equal(response.error, false);
        assert.equal(response.msg, 'Order already exists');        
    });

    it('orderExists: insert a null and recieve a null as response', function() {
        var extOrdID = null;
        var response = OrderCreateHelper.orderExists(extOrdID);

        assert.equal(response, null);
    });

    it('applyPriceAdjustments: add price adjustment', function() {
        var defaultPromoID = 'defaultPromoID';
        var priceAdjs = [
            {
                net_price: 1000,
                promotion_id: 'promoID'
            }
        ];
        var li = {
            createPriceAdjustment: function(val) {
                return {
                    value: val,
                    setPriceValue: function() {
                        return;
                    },
                    setTax: function (){
                        return;
                    },
                    updateTax: function (){
                        return;
                    },
                    updateTaxAmount: function (){
                        return;
                    }
                }
            }
        }
        var basket = new Basket();

        var testFn = OrderCreateHelper.applyPriceAdjustments;
        expect(function(){testFn(defaultPromoID, priceAdjs, li, basket)}).to.not.throw();
    });

    it('setShippingPriceAdjustments: set shipping price adjustments', function() {
       
        var shippingLineItem = {
            setPriceValue: function(){
                return
            },
            setTax: function(){
                return
            },
            updateTax: function(){
                return
            },
            updateTaxAmount: function(){
                return
            },
            createShippingPriceAdjustment: function(){
                return {
                    setPriceValue: function(){
                        return
                    },
                    setTax: function(){
                        return
                    },
                    updateTax: function(){
                        return
                    },
                    updateTaxAmount: function(){
                        return
                    }
                }
            },
        };
        var shippingTotalsJson = {
            original_shipping_fee: '10',
            tax: '10',
            shipping_fee_seller_discount: '10',
            shipping_fee_platform_discount: '10'
        };
        var order = {
            getCurrencyCode: function() {
                return;
            },
            updateTotals: function() {
                return;
            },
            updateOrderLevelPriceAdjustmentTax: function() {
                return;
            }
        };

        var testFn = OrderCreateHelper.setShippingPriceAdjustments;
        expect(function(){testFn(shippingLineItem, shippingTotalsJson, order)}).to.not.throw();
    });

});