'use strict';

/**
 * Generous JSON Product Feed Export
 * 
 * Exports product catalog data in Generous Product Feed Specification format (MVP v1.2)
 * Output: JSON file compatible with Generous Product API
 */

var File = require('dw/io/File');
var FileWriter = require('dw/io/FileWriter');
var Site = require('dw/system/Site');
var ProductMgr = require('dw/catalog/ProductMgr');
var CatalogMgr = require('dw/catalog/CatalogMgr');
var ProductSearchModel = require('dw/catalog/ProductSearchModel');
var Money = require('dw/value/Money');
var Logger = require('dw/system/Logger').getLogger('GenerousExport', 'GenerousExport');
var Status = require('dw/system/Status');
var Calendar = require('dw/util/Calendar');
var StringUtils = require('dw/util/StringUtils');
var URLUtils = require('dw/web/URLUtils');
var CustomObjectMgr = require('dw/object/CustomObjectMgr');

/**
 * Main export function - called by job step
 * @param {Object} parameters - Job parameters
 * @param {Object} stepExecution - Step execution context
 * @returns {dw.system.Status} - Job status
 */
function execute(parameters, stepExecution) {
    var feedConfigId = parameters.FeedConfigId || 'generous-feed';
    var hostname = parameters.Hostname || Site.current.getCustomPreferenceValue('hostname') || '';
    
    try {
        // Get feed configuration
        var feedConfig = CustomObjectMgr.getCustomObject('SalesChannelFeedConfig', feedConfigId);
        if (!feedConfig) {
            Logger.error('Feed configuration not found: ' + feedConfigId);
            return new Status(Status.ERROR, 'ERROR', 'Feed configuration not found: ' + feedConfigId);
        }
        
        // Get merchant ID - priority: Job Parameter > SitePreference > Feed Config
        var merchantId = parameters.MerchantId 
            || Site.current.getCustomPreferenceValue('generousMerchantId')
            || (feedConfig.custom.generousMerchantId ? feedConfig.custom.generousMerchantId : '');
        
        // Get merchant name - priority: Job Parameter > SitePreference > Site Name
        var merchantName = parameters.MerchantName 
            || Site.current.getCustomPreferenceValue('generousMerchantName')
            || Site.current.name || '';
        
        // Get default delivery days - priority: Job Parameter > SitePreference > Default
        var siteDeliveryDays = Site.current.getCustomPreferenceValue('generousDefaultDeliveryDays');
        var defaultDeliveryDays = parameters.DefaultDeliveryDays 
            ? parseInt(parameters.DefaultDeliveryDays, 10) 
            : (siteDeliveryDays ? parseInt(siteDeliveryDays, 10) : 5);
        
        var defaultCurrency = Site.current.defaultCurrency || 'USD';
        
        if (!merchantId) {
            Logger.error('Merchant ID is required. Set generousMerchantId in Site Preferences or feed configuration.');
            return new Status(Status.ERROR, 'ERROR', 'Merchant ID is required. Configure in Site Preferences > Generous Integration');
        }
        
        // Create output folder and file
        var folderPath = feedConfig.custom.folderName || 'IMPEX/src/feeds/export/social/generous/product';
        var folder = new File(folderPath);
        if (!folder.exists()) {
            folder.mkdirs();
        }
        
        var timestamp = StringUtils.formatCalendar(new Calendar(), 'yyyyMMddHHmmss');
        var siteId = Site.current.ID;
        var fileName = 'generous-feed-' + siteId + '-' + timestamp + '.json';
        var file = new File(folder, fileName);
        var fileWriter = new FileWriter(file, 'UTF-8');
        
        // Build feed structure
        var feed = {
            merchant: {
                id: merchantId,
                name: merchantName,
                default_estimated_delivery_days: defaultDeliveryDays
            },
            generated_at: new Calendar().time.toISOString(),
            default_currency: defaultCurrency,
            products: []
        };
        
        // Get products
        var products = getProducts(feedConfig, hostname);
        feed.products = products;
        
        // Write JSON to file
        var jsonString = JSON.stringify(feed, null, 2);
        fileWriter.write(jsonString);
        fileWriter.close();
        
        Logger.info('Generous feed exported successfully: ' + file.fullPath);
        Logger.info('Total products exported: ' + products.length);
        
        return new Status(Status.OK, 'OK', 'Exported ' + products.length + ' products to ' + fileName);
        
    } catch (e) {
        Logger.error('Error exporting Generous feed: ' + e.message + '\n' + e.stack);
        return new Status(Status.ERROR, 'ERROR', 'Export failed: ' + e.message);
    }
}

/**
 * Get products for export
 * @param {dw.object.CustomObject} feedConfig - Feed configuration
 * @param {string} hostname - Hostname for URLs
 * @returns {Array} - Array of product objects
 */
function getProducts(feedConfig, hostname) {
    var products = [];
    var includeOffline = feedConfig.custom.includeOfflineProducts || false;
    var includeOutOfStock = feedConfig.custom.includeOutOfStockProducts || true;
    var categoryId = feedConfig.custom.exportCategoryId || 'root';
    
    var siteCatalog = CatalogMgr.getSiteCatalog();
    var rootCategory = categoryId === 'root' ? siteCatalog.getRoot() : CatalogMgr.getCategory(categoryId);
    
    if (!rootCategory) {
        Logger.warn('Category not found: ' + categoryId + ', using root');
        rootCategory = siteCatalog.getRoot();
    }
    
    // Use ProductSearchModel to get searchable products
    var psm = new ProductSearchModel();
    psm.setCategoryID(rootCategory.ID);
    psm.setRecursiveCategorySearch(true);
    psm.search();
    
    var productIterator = psm.getProductSearchHits();
    var processedIds = {};
    
    while (productIterator.hasNext()) {
        var hit = productIterator.next();
        var product = hit.product;
        
        // Skip if already processed (avoid duplicates)
        if (processedIds[product.ID]) {
            continue;
        }
        processedIds[product.ID] = true;
        
        // Get the master or the product itself
        var masterProduct = product.isVariant() ? product.getMasterProduct() : product;
        
        // Skip if already processed master
        if (processedIds[masterProduct.ID]) {
            continue;
        }
        processedIds[masterProduct.ID] = true;
        
        // Skip offline products if configured
        if (!includeOffline && !masterProduct.online) {
            continue;
        }
        
        // Build product object
        var productObj = buildProductObject(masterProduct, hostname, includeOutOfStock);
        if (productObj) {
            products.push(productObj);
        }
    }
    
    return products;
}

/**
 * Build product object according to Generous spec
 * @param {dw.catalog.Product} product - SFCC Product
 * @param {string} hostname - Hostname for URLs
 * @param {boolean} includeOutOfStock - Include out of stock products
 * @returns {Object|null} - Product object or null if invalid
 */
function buildProductObject(product, hostname, includeOutOfStock) {
    try {
        var availability = getAvailability(product);
        
        // Skip out of stock if not configured to include
        if (!includeOutOfStock && availability === 'out_of_stock') {
            return null;
        }
        
        var price = getPrice(product);
        if (!price) {
            Logger.warn('No price for product: ' + product.ID);
            return null;
        }
        
        var imageUrl = getImageUrl(product, hostname);
        var productUrl = getProductUrl(product, hostname);
        
        var productObj = {
            // Required fields
            id: product.ID,
            title: product.name || '',
            description: product.shortDescription ? product.shortDescription.markup : (product.longDescription ? product.longDescription.markup : ''),
            product_url: productUrl,
            image_url: imageUrl,
            price: price,
            availability: availability,
            enable_search: product.searchable !== false ? 'true' : 'false',
            enable_checkout: product.online && availability !== 'out_of_stock' ? 'true' : 'false',
            
            // Recommended fields
            sku: product.manufacturerSKU || product.ID,
            brand: product.brand || '',
            tags: getTags(product),
            merchant_category: getCategoryPath(product),
            is_digital: !product.custom.requiresShipping,
            requires_shipping: product.custom.requiresShipping !== false,
            is_gift_card: product.giftCertificate || false
        };
        
        // Add Google product category if available
        if (product.custom.googleProductCategory) {
            productObj.google_product_category = product.custom.googleProductCategory;
        }
        
        // Add gift occasion hints if available
        if (product.custom.giftOccasionHints) {
            productObj.gift_occasion_hints = product.custom.giftOccasionHints;
        }
        
        // Add recipient tags if available
        if (product.custom.recipientTags) {
            productObj.recipient_tags = product.custom.recipientTags;
        }
        
        // Add estimated delivery days if available
        if (product.custom.estimatedDeliveryDays) {
            productObj.estimated_delivery_days = parseInt(product.custom.estimatedDeliveryDays, 10);
        }
        
        // Add variants if product has variations
        if (product.isMaster() && product.variants.length > 0) {
            productObj.variants = getVariants(product, hostname);
        }
        
        return productObj;
        
    } catch (e) {
        Logger.error('Error building product object for ' + product.ID + ': ' + e.message);
        return null;
    }
}

/**
 * Get product availability status
 * @param {dw.catalog.Product} product - SFCC Product
 * @returns {string} - Availability enum value
 */
function getAvailability(product) {
    var availabilityModel = product.availabilityModel;
    
    if (!availabilityModel) {
        return 'out_of_stock';
    }
    
    if (availabilityModel.isInStock()) {
        return 'in_stock';
    }
    
    var inventoryRecord = availabilityModel.inventoryRecord;
    if (inventoryRecord) {
        if (inventoryRecord.preorderable) {
            return 'preorder';
        }
        if (inventoryRecord.backorderable) {
            return 'backorder';
        }
    }
    
    return 'out_of_stock';
}

/**
 * Get price object
 * @param {dw.catalog.Product} product - SFCC Product
 * @returns {Object|null} - Price object with value and currency
 */
function getPrice(product) {
    var priceModel = product.priceModel;
    if (!priceModel) {
        return null;
    }
    
    var price = priceModel.price;
    if (!price || !price.available) {
        // Try to get min price for master products
        if (product.isMaster()) {
            price = priceModel.minPrice;
        }
    }
    
    if (!price || !price.available) {
        return null;
    }
    
    return {
        value: price.value.toFixed(2),
        currency: price.currencyCode
    };
}

/**
 * Get main product image URL
 * @param {dw.catalog.Product} product - SFCC Product
 * @param {string} hostname - Hostname for URLs
 * @returns {string} - Image URL
 */
function getImageUrl(product, hostname) {
    var image = product.getImage('large');
    if (!image) {
        image = product.getImage('medium');
    }
    if (!image) {
        image = product.getImage('small');
    }
    
    if (image) {
        var imageUrl = image.absURL.toString();
        if (hostname && imageUrl.indexOf('http') !== 0) {
            imageUrl = 'https://' + hostname + imageUrl;
        }
        return imageUrl;
    }
    
    return '';
}

/**
 * Get product detail page URL
 * @param {dw.catalog.Product} product - SFCC Product
 * @param {string} hostname - Hostname for URLs
 * @returns {string} - Product URL
 */
function getProductUrl(product, hostname) {
    var url = URLUtils.abs('Product-Show', 'pid', product.ID).toString();
    
    if (hostname) {
        // Replace hostname if provided
        url = url.replace(/https?:\/\/[^\/]+/, 'https://' + hostname);
    }
    
    return url;
}

/**
 * Get product tags
 * @param {dw.catalog.Product} product - SFCC Product
 * @returns {Array} - Array of tag strings
 */
function getTags(product) {
    var tags = [];
    
    // Add product type
    if (product.primaryCategory) {
        tags.push(product.primaryCategory.displayName);
    }
    
    // Add searchable keywords if available
    if (product.pageKeywords) {
        var keywords = product.pageKeywords.split(',');
        keywords.forEach(function(keyword) {
            var trimmed = keyword.trim();
            if (trimmed && tags.indexOf(trimmed) === -1) {
                tags.push(trimmed);
            }
        });
    }
    
    return tags;
}

/**
 * Get category path
 * @param {dw.catalog.Product} product - SFCC Product
 * @returns {string} - Category path
 */
function getCategoryPath(product) {
    var category = product.primaryCategory || product.classificationCategory;
    if (!category) {
        return '';
    }
    
    var path = [];
    var current = category;
    
    while (current && !current.root) {
        path.unshift(current.displayName);
        current = current.parent;
    }
    
    return path.join(' > ');
}

/**
 * Get product variants
 * @param {dw.catalog.Product} product - Master product
 * @param {string} hostname - Hostname for URLs
 * @returns {Array} - Array of variant objects
 */
function getVariants(product, hostname) {
    var variants = [];
    var variantIterator = product.variants.iterator();
    
    while (variantIterator.hasNext()) {
        var variant = variantIterator.next();
        
        if (!variant.online) {
            continue;
        }
        
        var variantObj = {
            id: variant.ID,
            sku: variant.manufacturerSKU || variant.ID,
            title: getVariantTitle(variant),
            price: getPrice(variant),
            availability: getAvailability(variant),
            image_url: getImageUrl(variant, hostname) || getImageUrl(product, hostname),
            options: getVariantOptions(variant)
        };
        
        variants.push(variantObj);
    }
    
    return variants;
}

/**
 * Get variant display title
 * @param {dw.catalog.Product} variant - Variant product
 * @returns {string} - Variant title
 */
function getVariantTitle(variant) {
    var parts = [];
    var variationModel = variant.variationModel;
    
    if (variationModel) {
        var attrs = variationModel.productVariationAttributes;
        for (var i = 0; i < attrs.length; i++) {
            var attr = attrs[i];
            var value = variationModel.getVariationValue(variant, attr);
            if (value) {
                parts.push(value.displayValue || value.value);
            }
        }
    }
    
    return parts.join(' / ') || variant.name || variant.ID;
}

/**
 * Get variant options
 * @param {dw.catalog.Product} variant - Variant product
 * @returns {Object} - Options object
 */
function getVariantOptions(variant) {
    var options = {};
    var variationModel = variant.variationModel;
    
    if (variationModel) {
        var attrs = variationModel.productVariationAttributes;
        for (var i = 0; i < attrs.length; i++) {
            var attr = attrs[i];
            var value = variationModel.getVariationValue(variant, attr);
            if (value) {
                options[attr.attributeID] = value.displayValue || value.value;
            }
        }
    }
    
    return options;
}

module.exports.execute = execute;

