'use strict';

var Logger = require('dw/system/Logger').getLogger('ocapi-hooks', 'basket');
var TaxMgr = require('dw/order/TaxMgr');
var collections = require('*/cartridge/scripts/util/collections');
var Money = require('dw/value/Money');
var ProductLineItem = require('dw/order/ProductLineItem');

/**
 * @param {number} taxAmount - The desired amount to calculate the percentage to be applied to a basePrice
 * @param {Money} adjustedNetPrice - The net price of the product line item after applying all product-level adjustments
 * @returns {number} calculated percentage
 */
const calculateTaxRate = function (taxAmount, adjustedNetPrice) {
    var moneyObj = new Money(taxAmount, adjustedNetPrice.currencyCode);
    var percentage = (moneyObj.percentOf(adjustedNetPrice)) / 100;
    return percentage;
};

/**
 * @param {*} productExternalTaxes JSON with productId and product's Tax
 * @param {*} productUUID Product ID.
 * @return {Object} productId and product's Tax
 */
function findProduct(productExternalTaxes, productUUID) {
    var productIdIdx = productExternalTaxes.map(function (pet) { return pet.itemId; }).indexOf(String(productUUID));
    return productIdIdx === -1 ? null : productExternalTaxes[productIdIdx];
}

/**
 * @param {*} lineItem lineItem
 * @param {*} taxRate taxRate
 * @param {string} lineItemType The type of line item sent to determine the name of the property that holds the price adjustments is. e.g. 'shippingPriceAdjustments' or 'priceAdjustments'
 * @param {booelan} isOrder boolean value to sinalize if the object froma order
 */
function updateTaxPriceAdjustments(lineItem, taxRate, lineItemType, isOrder) {
    var taxToBeApplied = !isOrder ? taxRate : 0;
    var priceAdjustmentObject = lineItemType === 'shippingLineItem' ? 'shippingPriceAdjustments' : 'priceAdjustments';
    collections.forEach(lineItem[priceAdjustmentObject], function (priceAdjustment) {
        priceAdjustment.setTaxClassID(TaxMgr.customRateTaxClassID);
        priceAdjustment.updateTax(taxToBeApplied);// percentage
    });
}

/**
 * @param {*} basketORorder basketORorder
 */
function setOrderPromotionTaxes(basketORorder) {
    // Order-level promotions for Instagram do not apply discounts to taxes, so the value set for taxes will be 0
    if (basketORorder.priceAdjustments.length > 0) {
        collections.forEach(basketORorder.priceAdjustments, function (pa) {
            pa.setTaxClassID(TaxMgr.customRateTaxClassID);
            pa.updateTax(0);
        });
    }
}

const updateExternalTaxesProductLineItems = function (basket) {
    if (Object.hasOwnProperty.call(basket.custom, 'productExternalTaxes')) {
        if (basket.custom.productExternalTaxes) {
            var productExternalTaxes;
            try {
                productExternalTaxes = JSON.parse(basket.custom.productExternalTaxes);
            } catch (e) {
                Logger.error('JSON parse error: ' + e.fileName + ':' + e.lineNumber + basket.custom.productExternalTaxes);
                throw new Error('Unable to parse JSON: ' + e.message);
            }
            collections.forEach(basket.productLineItems, function (pli) {
                var productTax = findProduct(productExternalTaxes, pli.UUID);
                if (productTax) {
                    var calculatedTaxRate = calculateTaxRate(productTax.tax, pli.adjustedNetPrice);
                    pli.setTaxClassID(TaxMgr.customRateTaxClassID);
                    pli.updateTax(calculatedTaxRate);// percentage
                    updateTaxPriceAdjustments(pli, calculatedTaxRate, 'productLineItem');
                    pli.custom.externalTax = productTax.tax;
                }
            });
        }

        setOrderPromotionTaxes(basket);
    }
};

const updateExternalTaxesLineItems = function (order, orderInput) {
    if (Object.hasOwnProperty.call(order.custom, 'externalTaxes')) {
        if (order.custom.externalTaxes) {
            var externalTaxes = orderInput.c_externalTaxes;

            collections.forEach(order.allLineItems, function (li) {
                var productTax = externalTaxes.get(li.UUID);
                if (productTax) {
                    var tax = productTax.tax;
                    var calculatedTaxRate = calculateTaxRate(tax, li.adjustedNetPrice);

                    li.setTaxClassID(TaxMgr.customRateTaxClassID);
                    li.setTax(new Money(tax, order.getCurrencyCode()));
                    li.updateTax(calculatedTaxRate);// percentage
                    li.updateTaxAmount(new Money(tax, order.getCurrencyCode()));

                    var lineItemType = li.class === ProductLineItem ? 'productLineItem' : 'shippingLineItem';
                    updateTaxPriceAdjustments(li, null, lineItemType, true);
                    li.custom.externalTax = productTax.tax;
                }
            });
        }

        setOrderPromotionTaxes(order);

        // Update totals and do not call calculate hook since it will override prices, promotions, and taxes
        order.updateTotals();
    }
};

module.exports = {
    calculateTaxRate: calculateTaxRate,
    updateExternalTaxesProductLineItems: updateExternalTaxesProductLineItems,
    updateTaxPriceAdjustments: updateTaxPriceAdjustments,
    updateExternalTaxesLineItems: updateExternalTaxesLineItems
};
