'use strict';

var Logger = require('dw/system/Logger');
var Order = require('dw/order/Order');
var Status = require('dw/system/Status');
var HookMgr = require('dw/system/HookMgr');

var OrderUtilCode = require('~/cartridge/scripts/util/OrderUtilCode');
var FixedPriceShippingDiscount = require('dw/campaign/FixedPriceShippingDiscount');
var FixedPriceDiscount = require('dw/campaign/FixedPriceDiscount');
var TaxMgr = require('dw/order/TaxMgr');
var Money = require('dw/value/Money');

var ReponseCode = OrderUtilCode.RESPONSE_CODE;

/**
 * add product line item price adjustment
 * @param {dw.order.Basket} order - current basket
 * @param {dw.order.LineItem} pli -product line item
 * */
function createProductAppeasement(order, pli) {
    var productLineItem = order.getProductLineItems(pli.productId).toArray()[0];

    if (productLineItem) {
        var currencyCode = order.getCurrencyCode();

        try {
            var priceAdjustment = productLineItem.createPriceAdjustment('appeaseament_' + pli.c_changeId, new FixedPriceDiscount(pli.c_productDiscount));
            priceAdjustment.setPriceValue(pli.c_productDiscount);
            priceAdjustment.setTaxClassID(TaxMgr.customRateTaxClassID);
            priceAdjustment.setTax(new Money(pli.c_taxDiscount, currencyCode));
            priceAdjustment.updateTaxAmount(new Money(pli.c_taxDiscount, currencyCode));

            var discount = new Money(pli.c_productDiscount, currencyCode);
            var taxDiscount = new Money(pli.c_taxDiscount, currencyCode);

            if (Object.hasOwnProperty.call(productLineItem, 'custom') && Object.hasOwnProperty.call(productLineItem.custom, 'externalLineItemAppeasementAmount')) {
                var currentExternalLineItemAppeasementAmount = new Money(productLineItem.custom.externalLineItemAppeasementAmount, currencyCode);
                var newExternalLineItemAppeasementAmount = discount.add(taxDiscount);
                productLineItem.custom.externalLineItemAppeasementAmount = currentExternalLineItemAppeasementAmount.add(newExternalLineItemAppeasementAmount).value;
            } else {
                productLineItem.custom.externalLineItemAppeasementAmount = discount.add(taxDiscount).value;
            }

            // Update totals and do not call calculate hook since it will override prices, promotions, and taxes
            order.updateTotals();
        } catch (error) {
            Logger.error('ERROR =' + error);
        }
    }
}

/**
 * add shipping line item shipping adjustment
 * @param {dw.order.Basket} order - current basket
 * @param {dw.order.Shipment} shippingLineItem -shipment line item
 * @param {dw.order.Shipment} shipmentCustom -shipment line item
 * */
function createShippingAppeasement(order, shippingLineItem, shipmentCustom) {
    var currencyCode = order.getCurrencyCode();

    try {
        var shippingCostRefund = shippingLineItem.createShippingPriceAdjustment('shipping_cost_appeasement_' + shipmentCustom.c_changeId, new FixedPriceShippingDiscount(shipmentCustom.c_shippingDiscount));
        shippingCostRefund.setPriceValue(shipmentCustom.c_shippingDiscount);
        shippingCostRefund.setTaxClassID(TaxMgr.customRateTaxClassID);
        shippingCostRefund.setTax(new Money(shipmentCustom.c_taxDiscount, currencyCode));
        shippingCostRefund.updateTax(shipmentCustom.c_taxDiscount);
        shippingCostRefund.updateTaxAmount(new Money(shipmentCustom.c_taxDiscount, currencyCode));

        var discount = new Money(shipmentCustom.c_shippingDiscount, order.getCurrencyCode());
        var taxDiscount = new Money(shipmentCustom.c_taxDiscount, order.getCurrencyCode());

        if (Object.hasOwnProperty.call(shippingLineItem, 'custom') && Object.hasOwnProperty.call(shippingLineItem.custom, 'externalShippingLineItemRefundAmount')) {
            var currentExternalShippingLineItemAppeasementAmount = new Money(shippingLineItem.custom.externalShippingLineItemRefundAmount, order.getCurrencyCode());
            var newExternalShippingLineItemAppeasementAmount = discount.add(taxDiscount);
            shippingLineItem.custom.externalShippingLineItemRefundAmount = currentExternalShippingLineItemAppeasementAmount.add(newExternalShippingLineItemAppeasementAmount).value;
        } else {
            shippingLineItem.custom.externalShippingLineItemRefundAmount = discount.add(taxDiscount).value;
        }

        // Update totals and do not call calculate hook since it will override prices, promotions, and taxes
        order.updateTotals();
    } catch (error) {
        Logger.error('ERROR =' + error);
    }
}

/**
 * Update the product line items with the external line item status and carrier.
 * @param {dw.order.Order} order order
 * @param {Object} orderInput Order Input
 */
function updateProductLineItem(order, orderInput) {
    var plis = orderInput.productItems.toArray();
    var restockingFeeHelpers = require('*/cartridge/scripts/util/restockingFeeHelpers');
    plis.forEach(function (pli) {
        var orderLI = order.getProductLineItems(pli.product_id).toArray()[0];
        if (orderLI) {
            if (pli.c_externalLineItemStatus != null) {
                orderLI.custom.externalLineItemStatus = pli.c_externalLineItemStatus;
            }

            if (pli.c_externalLineItemReturnStatus != null) {
                orderLI.custom.externalLineItemReturnStatus = pli.c_externalLineItemReturnStatus;
            }

            if (pli.c_quantityReceived != null) {
                orderLI.custom.quantityReceived = pli.c_quantityReceived;
            }

            if (pli.c_quantityReturned != null) {
                orderLI.custom.quantityReturned = pli.c_quantityReturned;
            }

            if (pli.c_quantityRejected != null) {
                orderLI.custom.quantityRejected = pli.c_quantityRejected;
            }

            if (pli.c_quantityCanceled != null) {
                orderLI.custom.quantityCanceled = pli.c_quantityCanceled;
            }

            if (pli.c_reason != null) {
                orderLI.custom.reason = pli.c_reason;
            }

            if (!orderInput.c_hasMultishipping) {
                if (pli.c_carrier != null) {
                    orderLI.custom.carrier = pli.c_carrier;
                }

                if (pli.c_trackingNumber != null) {
                    orderLI.custom.trackingNumber = pli.c_trackingNumber;
                }
            }

            if (pli.c_returnShippingLabel != null) {
                orderLI.custom.returnShippingLabel = pli.c_returnShippingLabel;
            }

            if (pli.c_productDiscount != null && pli.c_taxDiscount != null) {
                // create appeasment
                createProductAppeasement(order, pli);
            }
        }
        if (pli.c_restockingFee != null) {
            restockingFeeHelpers.updateOrderWithRestockingFee(order, pli.c_restockingFee);
        }
    });
}

/**
 * Update the shipment line items with the external line item status and carrier.
 * @param {dw.order.Order} order order
 * @param {Array} slis shipment line items
 */
function updateShipmentItems(order, slis) {
    slis.forEach(function (sli) {
        var shipment = order.getShipments().toArray()[0];
        if (shipment) {
            var shippingLineItem = shipment.shippingLineItems[0];
            if (shippingLineItem && sli.c_shippingDiscount != null && sli.c_taxDiscount != null) {
                // create appeasment
                createShippingAppeasement(order, shippingLineItem, sli);
            }
        }
    });
}

/**
 * Update the order with the multishipping information.
 * @param {dw.order.Order} order Current Order
 * @param {Object} orderInput Order Input from payload
 */
function updateMultiShipmentInformation(order, orderInput) {
    var plis = orderInput.productItems.toArray();

    var multiShippingInformationJson = [];

    plis.forEach(function (pli) {
        var orderLI = order.getProductLineItems(pli.product_id).toArray()[0];

        if (pli.c_OMSShipmentId) {
            multiShippingInformationJson[pli.product_id] = multiShippingInformationJson[pli.product_id] ? multiShippingInformationJson[pli.product_id] : [];

            multiShippingInformationJson[pli.product_id].push({
                trackingNumber: pli.c_trackingNumber,
                carrier: pli.c_carrier,
                quantity: pli.c_shipmentQuantity,
                shipmentId: pli.c_OMSShipmentId
            });

            orderLI.custom.multiShipmentInfo = JSON.stringify(multiShippingInformationJson[pli.product_id]);
        }
    });
}

/**
 * PATCH    /orders/{order_no}
 * Considered fields for update are status (same status transitions are possible as for
 * dw.order.Order.setStatus(int status) plus CREATED to FAILED) and custom properties.
 * @param {dw.order.Order} order order
 * @param {Object} orderInput orderInput
 * @returns {dw.system.Status} Status Status
 */
exports.afterPATCH = function (order, orderInput) {
    var response;
    // check if this is a social order (TikTok)
    if (order.getChannelType().value === order.CHANNEL_TYPE_TIKTOK) {
        if (!empty(orderInput) && !empty(orderInput.c_orderAction) && !empty(order.custom.orderAction)) {
            if (order.custom.orderAction.equalsIgnoreCase('return')) {
                if (HookMgr.callHook('app.order.return.processReturn', 'processReturn', order, orderInput)) {
                    response = ReponseCode.ORDERRETURN;
                    // update history
                } else {
                    Logger.error('ERROR processing return for Order =' + order.getOrderNo());
                    response = ReponseCode.ERROR_ORDERRETURN;
                }
            } else if (order.custom.orderAction.equalsIgnoreCase('return_order')) {
                if (HookMgr.callHook('app.order.return.processReturnOrder', 'processReturnOrder', order, orderInput)) {
                    response = ReponseCode.ORDERRETURN;
                    // update history
                } else {
                    Logger.error('ERROR processing return for Order =' + order.getOrderNo());
                    response = ReponseCode.ERROR_ORDERRETURN;
                }
            } else if (order.custom.orderAction.equalsIgnoreCase('cancel')) {
                if (HookMgr.callHook('app.order.cancel.processCancel', 'processCancel', order, orderInput)) {
                    response = ReponseCode.ORDERCANCEL_ITEMS;
                    // update history
                } else {
                    Logger.error('ERROR processing cancellation for items on Order =' + order.getOrderNo());
                    response = ReponseCode.ERROR_ORDERCANCEL_ITEMS;
                }
            } else if (order.custom.orderAction.equalsIgnoreCase('cancel_order')) {
                if (HookMgr.callHook('app.order.cancel.processCancelOrder', 'processCancelOrder', order, orderInput)) {
                    response = ReponseCode.ORDERCANCEL;
                    // update history
                } else {
                    Logger.error('ERROR processing cancellation for Order =' + order.getOrderNo());
                    response = ReponseCode.ERROR_ORDERCANCEL;
                }
            } else if (order.custom.orderAction.equalsIgnoreCase('refund')) {
                if (HookMgr.callHook('app.order.refund.processRefund', 'processRefund', order, orderInput)) {
                    response = ReponseCode.ORDERREFUND_ITEMS;
                    // update history
                } else {
                    Logger.error('ERROR processing refund items for Order =' + order.getOrderNo());
                    response = ReponseCode.ERROR_ORDER_REFUND_ITEMS;
                }
            } else {
                // check if request has product line item updates
                if (orderInput.productItems != null) {
                    updateProductLineItem(order, orderInput);
                }

                // check if request has shipment line item updates
                if (orderInput.shippingItems != null) {
                    updateShipmentItems(order, orderInput.shippingItems.toArray());
                }

                if (orderInput.c_hasMultishipping) {
                    updateMultiShipmentInformation(order, orderInput);
                }

                // update shipping status
                if (orderInput.shippingStatus != null) {
                    if (orderInput.shippingStatus.equalsIgnoreCase('shipped')) {
                        order.shippingStatus = Order.SHIPPING_STATUS_SHIPPED;
                    } else if (orderInput.shippingStatus.equalsIgnoreCase('part_shipped')) {
                        order.shippingStatus = Order.SHIPPING_STATUS_PARTSHIPPED;
                    }
                }
                Logger.info('Patch social order');
                response = ReponseCode.PATCH_SUCCESS;
            }
        } else {
            Logger.info('Patch social order');
            response = ReponseCode.PATCH_SUCCESS;
        }
        // Send request to process sfcc hook
        if (HookMgr.hasHook('app.process.sfccHook') && (('shippingStatus' in orderInput && orderInput.shippingStatus !== null && orderInput.shippingStatus.equalsIgnoreCase('shipped')) || order.custom.externalChannelOrderStatus.value === 6)) {
            var hookResult = HookMgr.callHook('app.process.sfccHook', 'processOrderUpdatesHook', order);
            if (empty(hookResult) || hookResult.error === true) {
                Logger.error('ERROR processing sfcc hook on order =' + order.getOrderNo());
            }
        }
    } else if (order.getChannelType().value === order.CHANNEL_TYPE_INSTAGRAMCOMMERCE) {
        if (!empty(orderInput) && !empty(orderInput.c_orderAction) && !empty(order.custom.orderAction)) {
            if (order.custom.orderAction.equalsIgnoreCase('return')) {
                if (HookMgr.callHook('app.order.return.processReturnItems', 'processReturnItems', order, orderInput)) {
                    response = ReponseCode.ORDERRETURN;
                } else {
                    Logger.error('ERROR processing return for Order =' + order.getOrderNo());
                    response = ReponseCode.ERROR_ORDERRETURN;
                }
            } else if (order.custom.orderAction.equalsIgnoreCase('return_request')) {
                if (HookMgr.callHook('app.order.return.processReturnRequest', 'processReturnRequest', order, orderInput)) {
                    response = ReponseCode.ORDERRETURN;
                    // update history
                } else {
                    Logger.error('ERROR processing return request for Order =' + order.getOrderNo());
                    response = ReponseCode.ERROR_ORDERRETURN;
                }
            } else if (order.custom.orderAction.equalsIgnoreCase('cancel')) {
                if (HookMgr.callHook('app.order.cancel.processCancel', 'processCancel', order, orderInput)) {
                    response = ReponseCode.ORDERCANCEL_ITEMS;
                    // update history
                } else {
                    Logger.error('ERROR processing cancellation for items on Order =' + order.getOrderNo());
                    response = ReponseCode.ERROR_ORDERCANCEL_ITEMS;
                }
            } else if (order.custom.orderAction.equalsIgnoreCase('cancel_order')) {
                if (HookMgr.callHook('app.order.cancel.processCancelOrder', 'processCancelOrder', order, orderInput)) {
                    response = ReponseCode.ORDERCANCEL;
                    // update history
                } else {
                    Logger.error('ERROR processing cancellation for Order =' + order.getOrderNo());
                    response = ReponseCode.ERROR_ORDERCANCEL;
                }
            } else if (order.custom.orderAction.equalsIgnoreCase('refund')) {
                if (HookMgr.callHook('app.order.refund.processRefund', 'processRefund', order, orderInput)) {
                    response = ReponseCode.ORDERREFUND_ITEMS;
                    // update history
                } else {
                    Logger.error('ERROR processing refund items for Order =' + order.getOrderNo());
                    response = ReponseCode.ERROR_ORDER_REFUND_ITEMS;
                }
            } else if (order.custom.orderAction.equalsIgnoreCase('refund_order')) {
                if (HookMgr.callHook('app.order.refund.processRefundOrder', 'processRefundOrder', order, orderInput)) {
                    response = ReponseCode.ORDERREFUND;
                    // update history
                } else {
                    Logger.error('ERROR processing refund for Order =' + order.getOrderNo());
                    response = ReponseCode.ERROR_ORDER_REFUND;
                }
            } else if (order.custom.orderAction.equalsIgnoreCase('refund_shipping_cost')) {
                if (HookMgr.callHook('app.order.refund.processRefundShippingCostOrder', 'processRefundShippingCostOrder', order, orderInput)) {
                    response = ReponseCode.SHIPPINGREFUND_COST;
                    // update history
                } else {
                    Logger.error('ERROR processing refund for Order =' + order.getOrderNo());
                    response = ReponseCode.ERROR_SHIPPING_REFUND_COST;
                }
            } else if (order.custom.orderAction.equalsIgnoreCase('tax_update')) {
                var externalTaxesHelpers = require('*/cartridge/scripts/util/externalTaxesHelpers');
                externalTaxesHelpers.updateExternalTaxesLineItems(order, orderInput);
                response = ReponseCode.TAX_UPDATE_SUCCESS;
            } else if (order.custom.orderAction.equalsIgnoreCase('appeasement')) {
                if (HookMgr.callHook('app.order.refund.processAppeasementLineItems', 'processAppeasementLineItems', order, orderInput)) {
                    response = ReponseCode.APPEASEMENT_SUCCESS;
                    // update history
                } else {
                    Logger.error('ERROR creating appeasement for Order =' + order.getOrderNo());
                    response = ReponseCode.APPEASEMENT_ERROR;
                }
            } else {
                // check if request has product line item updates
                if (orderInput.productItems != null) {
                    updateProductLineItem(order, orderInput);
                }

                // check if request has shipment line item updates
                if (orderInput.shippingItems != null) {
                    updateShipmentItems(order, orderInput.shippingItems.toArray());
                }

                if (orderInput.c_hasMultishipping) {
                    updateMultiShipmentInformation(order, orderInput);
                }

                // update shipping status
                if (orderInput.shippingStatus != null) {
                    if (orderInput.shippingStatus.equalsIgnoreCase('shipped')) {
                        order.shippingStatus = Order.SHIPPING_STATUS_SHIPPED;
                    } else if (orderInput.shippingStatus.equalsIgnoreCase('part_shipped')) {
                        order.shippingStatus = Order.SHIPPING_STATUS_PARTSHIPPED;
                    }
                }
                Logger.info('Patch social order');
                response = ReponseCode.PATCH_SUCCESS;
            }
        } else {
            Logger.info('Patch social order');
            response = ReponseCode.PATCH_SUCCESS;
        }
    } else {
        // OOB PATCH handling
        Logger.info('Patch order');
        response = ReponseCode.PATCH_SUCCESS;
    }

    return new Status(response.status, response.code, response.msg);
};
