'use strict';

var HookMgr = require('dw/system/HookMgr');
var Logger = require('dw/system/Logger');

var orderHelpers = require('~/cartridge/scripts/helpers/orderHelpers');
var som = require('~/cartridge/scripts/som');

const DEFAULT_RETURN_REASON = 'Unknown';

/**
 *   check som order status and compare with sfcc order if there is enough qty to return
 *   @param {Object} omsOrder - som order object
 *   @param {Object} sfccOrder - sfcc order object
 *   @param {Object} returnItems - items to return
 *   @param {boolean} returnShippingCost - sinalizes if return shipping cost is necessary
 *   @return {boolean} - return true if there was a status change
 */
function getReturnOrderItems(omsOrder, sfccOrder, returnItems, returnShippingCost) {
    var items2Return = returnItems.keySet().toArray();
    var omsLIs = omsOrder.orderItems.orderItems;
    var returnJSON = null;
    var changeItemArr = [];
    var isError = false;
    omsLIs.forEach(function (omsLi) {
        items2Return.forEach(function (productID) {
            if (omsLi.sfccProductId === productID) {
                var itemReturn = returnItems.get(productID);
                // check if qty available for return
                if (omsLi.quantityAvailableToReturn >= itemReturn.get('quantity')) {
                    // add item to returnJSON
                    var returnItemObj = {};
                    returnItemObj.id = omsLi.orderItemSummaryId;
                    returnItemObj.quantity = itemReturn.get('quantity');
                    returnItemObj.reason = itemReturn.get('reason');
                    if (itemReturn.get('c_restockingFee')) {
                        returnItemObj.restockingFee = {
                            product2Id: omsLi.product2Id,
                            priceBookEntryId: omsLi.pricebookEntryId,
                            amount: itemReturn.get('c_restockingFee')
                        };
                    }
                    changeItemArr.push(returnItemObj);
                } else {
                    Logger.error('Not available qty to return');
                    isError = true;
                }
            }
        });
    });
    if (!isError && changeItemArr.length > 0) {
        returnJSON = {};
        returnJSON.summaryId = omsOrder.orderSummaryId;
        returnJSON.lineItems = changeItemArr;
        returnJSON.orderId = sfccOrder.orderNo;
        returnJSON.currencyCode = sfccOrder.getCurrencyCode();
        // eslint-disable-next-line
        returnJSON.returnShippingCost = returnShippingCost ? true : false;
    }
    return returnJSON;
}

/**
 *   return full order get all the product line items to return
 *   @param {Object} sfccOrder - sfcc order object
 *   @param {Object} reason - reason for cancellation
 *   @param {Object} restockingFee - restocking fee amout
 *   @return {HashMap} - items to cancel
 */
function createReturnedItems(sfccOrder, reason, restockingFee) {
    var HashMap = require('dw/util/HashMap');
    var returnedItems = new HashMap();

    var plis = sfccOrder.getProductLineItems().iterator();
    while (plis.hasNext()) {
        var pli = plis.next();
        if (!pli.isBonusProductLineItem()) {
            var itemReturned = new HashMap();
            itemReturned.put('quantity', pli.getQuantityValue());
            itemReturned.put('reason', reason);
            if (restockingFee && !empty(restockingFee)) {
                itemReturned.put('c_restockingFee', restockingFee);
            }
            returnedItems.put(pli.productID, itemReturned);
        }
    }
    return returnedItems;
}

/**
 *   Verify if the requested shipping cost to refund exists in the order. If yes, build an object with data to create a payload
 *   @param {Object} omsOrder - som order object
 *   @param {Object} sfccOrder - sfcc order object
 *   @param {Object} refundShippingCostItems - shipments with shipping cost to refund
 *   @param {boolean} onlyShippingLineItem - signals if it is only necessary to return the shipping line item
 *   @return {Object} - Object with shipping cost data to refund
 */
function getRefundORAppeasementShippingCostItems(omsOrder, sfccOrder, refundShippingCostItems, onlyShippingLineItem) {
    var shipments2Refund = refundShippingCostItems.keySet().toArray();
    // For now, it handles only one shipment/shipping item per order, but some approaches have been taken to minimize rework in the future when multi-shipments are developed.
    var shipment2Refund = shipments2Refund[0];
    var shippingItemArr = [];

    var omsShippings = omsOrder.shipping;
    var returnJSON = {};

    var orderShipments = sfccOrder.shipments.toArray();
    var shipmentIdIdx = orderShipments.map(function (shipment) { return shipment.ID; }).indexOf(String(shipment2Refund));
    var shippingMatch = shipmentIdIdx === -1 ? null : orderShipments[shipmentIdIdx];

    if (empty(shippingMatch)) {
        Logger.error('Shipment ID not found');
        returnJSON.isError = true;
        return returnJSON;
    }

    var inputAdjustmentType = refundShippingCostItems.get(shipment2Refund).get('type').toLowerCase();
    var adjustmentType;

    switch (inputAdjustmentType) {
        case 'percentage':
            adjustmentType = 'Percentage';
            break;
        case 'amountwithtax':
            adjustmentType = 'AmountWithTax';
            break;
        case 'amountwithouttax':
            adjustmentType = 'AmountWithoutTax';
            break;
        default:
            adjustmentType = null;
            break;
    }

    var amount = refundShippingCostItems.get(shipment2Refund).get('amount') < 0 ? refundShippingCostItems.get(shipment2Refund).get('amount') : null;

    shippingItemArr.push({
        orderItemSummaryId: omsShippings.deliveryCharge.orderItemSummaryId,
        reason: refundShippingCostItems.get(shipment2Refund).get('reason'),
        adjustmentType: adjustmentType,
        amount: amount
    });

    returnJSON.isError = false;
    returnJSON.lineItems = shippingItemArr;

    if (onlyShippingLineItem) {
        return returnJSON;
    }

    returnJSON.summaryId = omsOrder.id;
    returnJSON.sfccOrder = sfccOrder;
    returnJSON.shippingMatch = shippingMatch;

    return returnJSON;
}

/**
 *   Prepare request data for payload object
 *   @param {Object} omsOrder - som order object
 *   @param {Object} sfccOrder - sfcc order object
 *   @param {Object} appeasementItems - items to create appeasement price adjustment
 *   @return {Object} - return true if there was a status change
 */
function getAppeasementItems(omsOrder, sfccOrder, appeasementItems) {
    var items2appeasement = appeasementItems.keySet().toArray();
    var omsLIs = omsOrder.orderLineItems;
    var appeasementJSON = {};
    var changeItemArr = [];
    var isError = false;
    omsLIs.forEach(function (omsLi) {
        items2appeasement.forEach(function (productID) {
            if (omsLi.ProductCode === productID) {
                var itemAppeasement = appeasementItems.get(productID);
                var inputAdjustmentType = itemAppeasement.get('type').toLowerCase();
                var adjustmentType;

                switch (inputAdjustmentType) {
                    case 'percentage':
                        adjustmentType = 'Percentage';
                        break;
                    case 'amountwithtax':
                        adjustmentType = 'AmountWithTax';
                        break;
                    case 'amountwithouttax':
                        adjustmentType = 'AmountWithoutTax';
                        break;
                    default:
                        adjustmentType = null;
                        isError = true;
                        break;
                }

                var amount;
                if (itemAppeasement.get('amount') < 0) {
                    amount = itemAppeasement.get('amount');
                } else {
                    isError = true;
                }

                // add item to appeasementJSON
                var itemAppeasementObj = {};
                itemAppeasementObj.orderItemSummaryId = omsLi.Id;
                itemAppeasementObj.reason = itemAppeasement.get('reason');
                itemAppeasementObj.adjustmentType = adjustmentType;
                itemAppeasementObj.amount = amount;
                changeItemArr.push(itemAppeasementObj);
            }
        });
    });

    if (isError || changeItemArr.length === 0) {
        appeasementJSON.isError = true;
        return appeasementJSON;
    }

    appeasementJSON = {};
    appeasementJSON.summaryId = omsOrder.id;
    appeasementJSON.lineItems = changeItemArr;
    appeasementJSON.orderId = sfccOrder.orderNo;
    appeasementJSON.currencyCode = sfccOrder.getCurrencyCode();

    return appeasementJSON;
}

/**
*   refund the given items by calling SOM API
*   @param {Object} order - sfcc order object
*   @param {Object} orderInput -  JSON of which items needs to be refund
*   @return {boolean} - return true if no API exceptions
*/
exports.processRefund = function (order, orderInput) {
    var orderID = order.getOrderNo();
    var somOrder = orderHelpers.getOrderSummary([orderID]);
    // make sure the OMS order is not shipped
    if (somOrder != null && somOrder.length > 0 && somOrder[0].shippedStatusGroupItems != null && somOrder[0].shippedStatusGroupItems.length > 0) {
        if (order.custom.returnCase != null && order.custom.returnCase.indexOf('returnItems') > 0) {
            var returnItems = orderInput.c_returnCase.get('returnItems');
            var returnData = getReturnOrderItems(somOrder[0].shippedStatusGroupItems[0], order, returnItems);
            if (returnData != null) {
                var somRes = som.returnOrderItems(returnData);
                if (somRes.ok && !somRes.object.isError) {
                    // update SFCC order status
                    var HashMap = require('dw/util/HashMap');
                    var orderNumbers = [];
                    var sfccOrders = new HashMap();
                    orderNumbers.push(orderID);
                    sfccOrders.put(orderID, order);
                    HookMgr.callHook('app.order.update.processStatusUpdate', 'processStatusUpdate', JSON.stringify(orderNumbers), sfccOrders);
                    return true;
                }
                Logger.error('Error refunding order items in SOM');
                return false;
            }
        } else {
            Logger.error('No items to refund in the request');
            return false;
        }
    }
    Logger.error('Couldn\'t retrieve shipped items from SOM or items not shipped yet; can\'t be returned/refunded');
    return false;
};

/**
*   refund order by calling SOM API
*   @param {Object} order - sfcc order object
*   @param {Object} orderInput -  JSON of which items needs to be refunded
*   @return {boolean} - return true if no API exceptions
*/
exports.processRefundOrder = function (order, orderInput) {
    var HashMap = require('dw/util/HashMap');
    var orderID = order.getOrderNo();
    var somOrder = orderHelpers.getOrderSummary([orderID]);
    var sfccOrders;

    // make sure the OMS order is shipped
    if (somOrder != null && somOrder.length > 0 && somOrder[0].shippedStatusGroupItems != null && somOrder[0].shippedStatusGroupItems.length > 0) {
        var returnReason = orderInput.c_returnCase.get('reason');
        var restockingFee = orderInput.c_returnCase.get('c_restockingFee');
        if (returnReason == null) {
            returnReason = DEFAULT_RETURN_REASON;
        }
        var returnItems = createReturnedItems(order, returnReason, restockingFee);
        var returnData = getReturnOrderItems(somOrder[0].shippedStatusGroupItems[0], order, returnItems, true);

        if (returnData != null) {
            var somRes = som.returnOrderItems(returnData);
            if (somRes.ok && !somRes.object.isError) {
                // update SFCC order status
                var orderNumbers = [];
                sfccOrders = new HashMap();
                orderNumbers.push(orderID);
                sfccOrders.put(orderID, order);
                HookMgr.callHook('app.order.update.processStatusUpdate', 'processStatusUpdate', JSON.stringify(orderNumbers), sfccOrders);
                return true;
            }
            Logger.error('Error creating return order items for refund in SOM');
            return false;
        }
    }
    Logger.error('Couldn\'t retrieve shipped items from SOM or items not shipped yet; can\'t be returned/refunded');
    return false;
};

/**
*   refund shipping cost by calling SOM API
*   @param {Object} order - sfcc order object
*   @param {Object} orderInput -  JSON of which items needs to be refunded
*   @return {boolean} - return true if no API exceptions
*/
exports.processRefundShippingCostOrder = function (order, orderInput) {
    var orderID = order.getOrderNo();
    var somOrder = orderHelpers.getOrderSummary([orderID]);
    var refundShippingCostItemsData;

    // make sure the OMS order is not shipped
    if (somOrder != null && somOrder.length > 0 && somOrder[0].shippedStatusGroupItems != null && somOrder[0].shippedStatusGroupItems.length > 0) {
        if (order.custom.returnCase != null && order.custom.returnCase.indexOf('shippingItems') > 0) {
            if (order.custom.returnCase.indexOf('shippingItems') > 0) {
                var refundShippingCostItems = orderInput.c_returnCase.get('shippingItems');
                refundShippingCostItemsData = getRefundORAppeasementShippingCostItems(somOrder[0], order, refundShippingCostItems, false);

                if (!refundShippingCostItemsData.isError) {
                    var somAdjustmentRes = som.createAdjustmentForShippingCostRefund(refundShippingCostItemsData);
                    if (!somAdjustmentRes.ok) {
                        Logger.error('Error refunding shipping cost in SOM');
                        return false;
                    }
                    somOrder = orderHelpers.getOrderSummary([orderID]);
                } else {
                    Logger.error('No items to refund in the request');
                    return false;
                }
            }
            return true;
        }
    }
    Logger.error('Couldn\'t retrieve shipped items from SOM or items not shipped yet; can\'t be returned/refunded');
    return false;
};

/**
*   refund shipping cost by calling SOM API
*   @param {Object} order - sfcc order object
*   @param {Object} orderInput -  JSON of which items needs to be refunded
*   @return {boolean} - return true if no API exceptions
*/
exports.processAppeasementLineItems = function (order, orderInput) {
    var orderID = order.getOrderNo();
    var somOrder = orderHelpers.getOrderSummary([orderID]);
    var appeasementItemsData;
    var shipmentItemsData;
    var appeasementItemsAllData;

    if (order.custom.appeasement != null && (order.custom.appeasement.indexOf('appeasementItems') > 0 || order.custom.appeasement.indexOf('shippingItems'))) {
        var appeasementItems = orderInput.c_appeasement.get('appeasementItems');
        var shipmentItems = orderInput.c_appeasement.get('shippingItems');
        if (appeasementItems) {
            appeasementItemsData = getAppeasementItems(somOrder[0], order, appeasementItems);
            if (appeasementItemsData.isError) {
                Logger.error('Error creating payload for adjustment in SOM');
                return false;
            }
            appeasementItemsAllData = appeasementItemsData;
            if (shipmentItems) {
                shipmentItemsData = getRefundORAppeasementShippingCostItems(somOrder[0], order, shipmentItems, true);
                if (shipmentItemsData.isError) {
                    Logger.error('Error creating payload for adjustment in SOM');
                    return false;
                }
                appeasementItemsAllData.lineItems = appeasementItemsAllData.lineItems.concat(shipmentItemsData.lineItems);
            }
        } else {
            shipmentItemsData = getRefundORAppeasementShippingCostItems(somOrder[0], order, shipmentItems, false);
            if (shipmentItemsData.isError) {
                Logger.error('Error creating payload for adjustment in SOM');
                return false;
            }
            appeasementItemsAllData = shipmentItemsData;
        }

        var somAdjustmentRes = som.createAdjustmentForAppeasement(appeasementItemsAllData);
        if (!somAdjustmentRes.ok) {
            Logger.error('Error creating adjustment price for appeasement in SOM');
            return false;
        }

        return true;
    }

    Logger.error('No items to appeasement in the request');
    return false;
};
