'use strict';

var Order = require('dw/order/Order');
var Logger = require('dw/system/Logger');
var OrderUtilCode = require('*/cartridge/scripts/util/OrderUtilCode');
var orderRefundHelpers = require('*/cartridge/scripts/helpers/orderRefundHelpers');
var TaxMgr = require('dw/order/TaxMgr');
var Money = require('dw/value/Money');
var FixedPriceShippingDiscount = require('dw/campaign/FixedPriceShippingDiscount');
var FixedPriceDiscount = require('dw/campaign/FixedPriceDiscount');

/**
*   refund the whole order
*   @param {Object} order - sfcc order object
*   @param {Object} orderInput -  JSON of which items needs to be refunded
*   @return {boolean} - return true if no API exceptions
*/
exports.processRefundOrder = function (order, orderInput) { // eslint-disable-line no-unused-vars
    // refund the entire orer
    order.custom.externalChannelOrderStatus = OrderUtilCode.EXTERNAL_ORDER_STATUS.REFUNDED;
    order.custom.externalReturnStatus = OrderUtilCode.EXTERNAL_RETURN_STATUS.REFUNDED;
    var producLineItems = order.getProductLineItems().iterator();
    while (producLineItems.hasNext()) {
        var currentProductLineItem = producLineItems.next();
        currentProductLineItem.custom.externalLineItemReturnStatus = OrderUtilCode.EXTERNAL_RETURN_STATUS.REFUNDED;
        currentProductLineItem.custom.quantityReturned = currentProductLineItem.quantity.value;
    }
    return true;
};

/**
*   refund the items specified for the order
*   @param {Object} order - sfcc order object
*   @param {Object} orderInput -  JSON of which items needs to be refunded
*   @return {boolean} - return true if no API exceptions
*/
exports.processRefund = function (order, orderInput) { // eslint-disable-line no-unused-vars
    if (order.custom.returnCase != null && order.custom.returnCase.indexOf('returnItems') > 0) {
        var returnItems = orderInput.c_returnCase.get('returnItems');
        // Refund the items specified for the order
        var items2Refund = returnItems.keySet().toArray();
        var plis = order.getProductLineItems().iterator();
        var insufficientQuantity = false;
        var insufficientItems = [];

        while (plis.hasNext()) {
            var pli = plis.next();
            if (!pli.isBonusProductLineItem()) {
                items2Refund.forEach(function (productID) { // eslint-disable-line no-loop-func
                    if (pli.productID === productID) {
                        var quantityToRefund = Number(returnItems.get(productID).get('quantity'));
                        var quantityAvailableToRefund = orderRefundHelpers.getQuantityAvailableToRefund(pli);
                        if (quantityToRefund <= quantityAvailableToRefund) {
                            if (orderRefundHelpers.isLineItemWillBeFullyRefunded(pli, quantityToRefund)) {
                                pli.custom.externalLineItemReturnStatus = OrderUtilCode.EXTERNAL_LINE_ITEM_RETURN_STATUS.REFUNDED;
                            } else {
                                pli.custom.externalLineItemReturnStatus = OrderUtilCode.EXTERNAL_LINE_ITEM_RETURN_STATUS.PARTIAL_REFUND;
                            }
                            orderRefundHelpers.saveQuantityRefunded(pli, quantityToRefund);
                        } else {
                            insufficientQuantity = true;
                            insufficientItems.push({ productID: productID, quantityAvailableToRefund: quantityAvailableToRefund, quantityRequestedToRefund: quantityToRefund });
                        }
                    }
                });
            }
        }

        if (insufficientQuantity) {
            Logger.error('items with quantity not available for refund: \n' + JSON.stringify(insufficientItems));
            return false;
        }

        if (orderRefundHelpers.isOrderFullyRefunded(order)) {
            order.custom.externalChannelOrderStatus = OrderUtilCode.EXTERNAL_RETURN_STATUS.REFUNDED;
            order.custom.externalReturnStatus = OrderUtilCode.EXTERNAL_RETURN_STATUS.REFUNDED;
        } else {
            order.custom.externalChannelOrderStatus = OrderUtilCode.EXTERNAL_RETURN_STATUS.PARTIAL_REFUND;
            order.custom.externalReturnStatus = OrderUtilCode.EXTERNAL_RETURN_STATUS.PARTIAL_REFUND;
        }

        order.custom.externalExportStatus = Order.EXPORT_STATUS_READY;
    } else {
        Logger.error('No items to refund in the request');
        return false;
    }
    return true;
};

/**
*   refund the shipment specified for the order
*   @param {Object} order - sfcc order object
*   @param {Object} orderInput -  JSON of which items needs to be refunded
*   @return {boolean} - return true if no API exceptions
*/
exports.processRefundShippingCostOrder = function (order, orderInput) { // eslint-disable-line no-unused-vars
    if (order.custom.returnCase != null && order.custom.returnCase.indexOf('shippingItems') > 0) {
        var returnItems = orderInput.c_returnCase.get('shippingItems');
        // Refund the items specified for the order
        var items2Refund = returnItems.keySet().toArray();
        var currencyCode = order.getCurrencyCode();

        items2Refund.forEach(function (shipment2refund) {
            var currentShipment = order.getShipment(shipment2refund);
            var currentShippinglineItem = currentShipment.getStandardShippingLineItem();
            var refundRequest = returnItems.get(shipment2refund);
            var discountAmount = Number(refundRequest.get('amount'));
            var timeStamp = new Date().getTime();

            var shippingCostRefund = currentShippinglineItem.createShippingPriceAdjustment('shipping_cost_appeasement_' + timeStamp, new FixedPriceShippingDiscount(discountAmount));
            shippingCostRefund.setPriceValue(discountAmount);
            shippingCostRefund.setTaxClassID(TaxMgr.customRateTaxClassID);
            shippingCostRefund.setTax(new Money(0, currencyCode));
            shippingCostRefund.updateTax(0);
            shippingCostRefund.updateTaxAmount(new Money(0, currencyCode));

            var discount = new Money(discountAmount, order.getCurrencyCode());
            var taxDiscount = new Money(0, order.getCurrencyCode());

            if (Object.hasOwnProperty.call(currentShippinglineItem, 'custom') && Object.hasOwnProperty.call(currentShippinglineItem.custom, 'externalShippingLineItemRefundAmount')) {
                var currentExternalShippingLineItemAppeasementAmount = new Money(currentShippinglineItem.custom.externalShippingLineItemRefundAmount, order.getCurrencyCode());
                var newExternalShippingLineItemAppeasementAmount = discount.add(taxDiscount);
                currentShippinglineItem.custom.externalShippingLineItemRefundAmount = currentExternalShippingLineItemAppeasementAmount.add(newExternalShippingLineItemAppeasementAmount).value;
            } else {
                currentShippinglineItem.custom.externalShippingLineItemRefundAmount = discount.add(taxDiscount).value;
            }

            // Update totals and do not call calculate hook since it will override prices, promotions, and taxes
            order.updateTotals();

            if (currentShippinglineItem.adjustedNetPrice.value <= 0) {
                currentShippinglineItem.custom.externalShippingLineItemRefundStatus = OrderUtilCode.OMS_SHIPPING_REFUND_STATUS.REFUNDED;
            } else {
                currentShippinglineItem.custom.externalShippingLineItemRefundStatus = OrderUtilCode.OMS_SHIPPING_REFUND_STATUS.PARTIAL_REFUND;
            }
        });
    } else {
        Logger.error('No items to refund in the request');
        return false;
    }
    return true;
};

/**
*   Creates price adjustments to represent the appeasements created for the items specified for the order
*   @param {Object} order - sfcc order object
*   @param {Object} orderInput -  JSON of which items needs to be refunded
*   @return {boolean} - return true if no API exceptions
*/
exports.processAppeasementLineItems = function (order, orderInput) { // eslint-disable-line no-unused-vars
    var items2appeasement = [];
    var shippingsItems2appeasement = [];

    var appeasementItems = orderInput.c_appeasement.get('appeasementItems');
    var shipmentItems = orderInput.c_appeasement.get('shippingItems');

    if (appeasementItems) {
        items2appeasement = appeasementItems.keySet().toArray();
    }

    if (shipmentItems) {
        shippingsItems2appeasement = shipmentItems.keySet().toArray();
    }

    if (shippingsItems2appeasement.length > 0) {
        shippingsItems2appeasement.forEach(function (shipment2appeasement) {
            var currentShipment = order.getShipment(shipment2appeasement);
            var currentShippinglineItem = currentShipment.getStandardShippingLineItem();

            var refundRequest = shipmentItems.get(shipment2appeasement);
            var discountAmount = Number(refundRequest.get('amount'));

            var shippingCostRefund = currentShippinglineItem.createShippingPriceAdjustment('shipping_cost_appeasement_' + (new Date()).toISOString(), new FixedPriceShippingDiscount(discountAmount));
            shippingCostRefund.setPriceValue(discountAmount);
            shippingCostRefund.setTaxClassID(TaxMgr.customRateTaxClassID);
            shippingCostRefund.setTax(new Money(0, order.getCurrencyCode()));
            shippingCostRefund.updateTax(0);
            shippingCostRefund.updateTaxAmount(new Money(0, order.getCurrencyCode()));

            var discount = new Money(discountAmount, order.getCurrencyCode());
            var taxDiscount = new Money(0, order.getCurrencyCode());

            if (Object.hasOwnProperty.call(currentShippinglineItem, 'custom') && Object.hasOwnProperty.call(currentShippinglineItem.custom, 'externalShippingLineItemRefundAmount')) {
                var currentExternalShippingLineItemAppeasementAmount = new Money(currentShippinglineItem.custom.externalShippingLineItemRefundAmount, order.getCurrencyCode());
                var newExternalShippingLineItemAppeasementAmount = discount.add(taxDiscount);

                currentShippinglineItem.custom.externalShippingLineItemRefundAmount = currentExternalShippingLineItemAppeasementAmount.add(newExternalShippingLineItemAppeasementAmount).value;
            } else {
                currentShippinglineItem.custom.externalShippingLineItemRefundAmount = discount.add(taxDiscount).value;
            }

            order.updateTotals();

            if (currentShippinglineItem.adjustedNetPrice.value <= 0) {
                currentShippinglineItem.custom.externalShippingLineItemRefundStatus = OrderUtilCode.OMS_SHIPPING_REFUND_STATUS.REFUNDED;
            } else {
                currentShippinglineItem.custom.externalShippingLineItemRefundStatus = OrderUtilCode.OMS_SHIPPING_REFUND_STATUS.PARTIAL_REFUND;
            }
        });
    }

    if (items2appeasement.length > 0) {
        items2appeasement.forEach(function (item2appeasement) {
            var pli = order.getProductLineItems(item2appeasement).toArray()[0];
            var refundRequest = appeasementItems.get(item2appeasement);
            var discountAmount = Number(refundRequest.get('amount'));

            var pliAppeasement = pli.createPriceAdjustment('appeaseament_' + (new Date()).toISOString(), new FixedPriceDiscount(discountAmount));
            pliAppeasement.setPriceValue(discountAmount);
            pliAppeasement.setTaxClassID(TaxMgr.customRateTaxClassID);
            pliAppeasement.setTax(new Money(0, order.getCurrencyCode()));
            pliAppeasement.updateTax(0);
            pliAppeasement.updateTaxAmount(new Money(0, order.getCurrencyCode()));
            pliAppeasement.custom.askedBy = 'meta';

            if (Object.hasOwnProperty.call(pli, 'custom') && Object.hasOwnProperty.call(pli.custom, 'externalLineItemAppeasementAmount')) {
                var currentExternalLineItemAppeasementAmount = new Money(pli.custom.externalLineItemAppeasementAmount, order.getCurrencyCode());
                var newExternalLineItemAppeasementAmount = currentExternalLineItemAppeasementAmount.add(new Money(discountAmount, order.getCurrencyCode()));
                pli.custom.externalLineItemAppeasementAmount = newExternalLineItemAppeasementAmount.value;
            } else {
                pli.custom.externalLineItemAppeasementAmount = discountAmount;
            }

            order.updateTotals();

            if (pli.adjustedNetPrice.value <= 0) {
                pli.custom.externalLineItemReturnStatus = OrderUtilCode.EXTERNAL_LINE_ITEM_RETURN_STATUS.REFUNDED;
            } else {
                pli.custom.externalLineItemReturnStatus = OrderUtilCode.EXTERNAL_LINE_ITEM_RETURN_STATUS.PARTIAL_REFUND;
            }
        });
    }

    if (order.totalGrossPrice.value <= 0) {
        order.custom.externalReturnStatus = OrderUtilCode.EXTERNAL_ORDER_STATUS.REFUNDED;
    } else {
        order.custom.externalReturnStatus = OrderUtilCode.EXTERNAL_ORDER_STATUS.PARTIAL_REFUND;
    }

    return true;
};
