'use strict';

var server = require('server');
server.extend(module.superModule);

var Logger = require('dw/system/Logger').getLogger('int_google_checkout', 'CartLink');
var LinkHelpers = require('*/cartridge/scripts/utils/LinkHelpers.js');

/**
 * Add products and your quantities to current basket
 */
server.replace('Cart', function (req, res, next) {
    var BasketMgr = require('dw/order/BasketMgr');
    var Transaction = require('dw/system/Transaction');
    var cartHelper = require('*/cartridge/scripts/cart/cartHelpers');
    var URLUtils = require('dw/web/URLUtils');
    var ProductFactory = require('*/cartridge/scripts/factories/product');

    // eslint-disable-next-line no-unneeded-ternary
    var checkout = (empty(req.querystring.checkout) || (req.querystring.checkout.toLowerCase() === 'false' || req.querystring.checkout.toLowerCase() !== 'true')) ? false : true;

    var action = res.viewData.action;

    var socialBasket;

    try {
        if (empty(action) || !LinkHelpers.isAValidActionString(action)) {
            throw new Error('cartlink.action.not.valid');
        }

        var pidsObj = LinkHelpers.createPidsObj(action);

        if (!empty(pidsObj)) {
            var tempSocialBasket = req.session.privacyCache.get('tempSocialBasket');
            if (!empty(tempSocialBasket) && !empty(BasketMgr.getTemporaryBasket(tempSocialBasket))) {
                socialBasket = BasketMgr.getTemporaryBasket(tempSocialBasket);
            } else {
                try {
                    Transaction.wrap(function () {
                        socialBasket = BasketMgr.createTemporaryBasket();
                        req.session.privacyCache.set('tempSocialBasket', socialBasket.UUID);
                    });
                } catch (error) {
                    if (error.type === 'CreateTemporaryBasketLimitExceededException') {
                        Transaction.wrap(function () {
                            var temporaryBasket = BasketMgr.getTemporaryBaskets()[0];
                            BasketMgr.deleteTemporaryBasket(temporaryBasket);
                            socialBasket = BasketMgr.createTemporaryBasket();
                            req.session.privacyCache.set('tempSocialBasket', socialBasket.UUID);
                        });
                    } else {
                        throw error;
                    }
                }
            }

            if (!empty(socialBasket)) {
                Transaction.wrap(function () {
                    socialBasket.setChannelType(socialBasket.CHANNEL_TYPE_GOOGLE);

                    pidsObj.forEach(function (PIDObj) {
                        var product = ProductFactory.get({ pid: PIDObj.pid });
                        var quantity = PIDObj.qty;
                        var options = !empty(product.options) ? product.options[0] : {};
                        if (product.productType !== 'master' && product.productType !== 'variationGroup' && product.productType !== 'set') {
                            var PIDObjResult = cartHelper.addProductToCart(
                                socialBasket,
                                product.id,
                                quantity,
                                [],
                                options
                            );
                            if (PIDObjResult.error) {
                                Logger.error('Error occurred while {0}. Error Message: {1}', PIDObjResult.error, PIDObjResult.message);
                            }
                        }
                    });
                });

                req.session.privacyCache.set('checkout', checkout);

                if (!req.currentCustomer.raw.authenticated) {
                    req.session.privacyCache.set('googleCartLink', JSON.stringify(LinkHelpers.generateGooglePermLinks(socialBasket, checkout)));
                }

                if (checkout) {
                    res.redirect(URLUtils.url('Checkout-Begin'));
                    return next();
                }
            } else {
                throw new Error('cartlink.temp.basket.error.create');
            }
        }

        res.redirect(URLUtils.url('Cart-Show'));
    } catch (e) {
        Logger.error(e.message);
        res.redirect(URLUtils.url('Error-ErrorCode', 'err', e.message));
    }

    return next();
});

module.exports = server.exports();
