'use strict';

var ArrayList = require('dw/util/ArrayList');
var PropertyComparator = require('dw/util/PropertyComparator');
var Resource = require('dw/web/Resource');
var Site = require('dw/system/Site');
var StringUtils = require('dw/util/StringUtils');
var System = require('dw/system/System');
var Transaction = require('dw/system/Transaction');
var Logger = require('dw/system/Logger').getLogger('Snapchat', 'snapchatService');

var serviceHelper = require('./serviceHelper');
var constants = require('../SnapchatConstants');
var customObjectHelper = require('../customObjectHelper');

var counter = 0;

/**
 * Parses the response and trigger the given {callback} in case of success or redirect to the landing page in case of error
 * Snapchat ServiceCredential
 * @param {Object} result The result of the response
 * @param {string} errorCode The error code from the response
 * @returns {Object} result object
 */
function parseResponse(result, errorCode) {
    var Result = require('dw/svc/Result');

    var snapchatSettings = customObjectHelper.getCustomObject();

    var accessToken = snapchatSettings.custom.accessToken || null;
    var refreshToken = snapchatSettings.custom.refreshToken || null;

    if (!result.ok && result.error === 401) {
        if (refreshToken && snapchatSettings.custom.appId && snapchatSettings.custom.appSecret) {
            getRefreshToken(snapchatSettings.custom.appId, snapchatSettings.custom.appSecret, refreshToken); // eslint-disable-line no-use-before-define
        }
        return {
            error: true,
            errorCode: 'retry'
        };
    }

    if (!result.ok && result.status === Result.SERVICE_UNAVAILABLE) {
        return {
            error: true,
            errorCode: Result.SERVICE_UNAVAILABLE
        };
    }

    if (!result.ok && (!accessToken || !refreshToken)) {
        Logger.warn('snapchatSettings.custom.accessToken = {0}, snapchatSettings.custom.refreshToken = {1}', accessToken, refreshToken);

        // Fill form fields from custom object
        customObjectHelper.fillFormFromCustomObject(snapchatSettings);

        // Render the landing page so that the customer can authenticate through Snapchat
        require('dw/template/ISML').renderTemplate('snapchat/start', {
            acceptTerms: snapchatSettings.custom.acceptTerms,
            success: request.httpParameterMap.success.stringValue
        });
        return {
            error: true,
            errorCode: errorCode
        };
    }

    if (!result.ok && result.error !== '401') {
        Logger.error('Error occurred while {0}. Error Message: {1}', errorCode, result.errorMessage);
        return {
            error: true,
            errorCode: errorCode
        };
    }

    var resultText;
    try {
        resultText = JSON.parse(result.object.text);
        Logger.info(counter++ + ': result: ' + result.object.text + '\n');
    } catch (e) {
        Logger.error(e.toString() + ' in ' + e.fileName + ':' + e.lineNumber);
    }

    if (!resultText) {
        return {
            error: true,
            errorCode: errorCode
        };
    }

    if (result.ok && resultText) {
        if (resultText.access_token && resultText.refresh_token
            && (resultText.access_token !== accessToken || resultText.refresh_token !== refreshToken)) {
            Transaction.wrap(function () {
                snapchatSettings.custom.accessToken = resultText.access_token;
                snapchatSettings.custom.refreshToken = resultText.refresh_token;
            });
        }
        return {
            error: false,
            result: resultText
        };
    }
    Logger.error('Error occurred {0}.', errorCode);
    return {
        error: true,
        errorCode: errorCode
    };
}

/**
 * Get the refresh token from the Snapchat REST API
 * @param {string} appId The Snapchat OAuth App ID
 * @param {string} appSecret The Snapchat OAuth App Secret
 * @param {string} refreshToken The Snapchat refresh_token
 * @returns {Object} an object containing the error or response payload
 */
function getRefreshToken(appId, appSecret, refreshToken) {
    var errorCode = 'oauth.refresh_token.call';
    var service = serviceHelper.getService(constants.SERVICES.SNAPCHAT_ACCOUNTS);
    var params = {
        action: constants.ACTIONS.GET_REFRESH_TOKEN,
        method: 'POST',
        path: constants.ENDPOINTS.REFRESH,
        params: {
            client_id: appId,
            client_secret: appSecret,
            refresh_token: refreshToken,
            grant_type: 'refresh_token'
        }
    };
    var result = service.call(params);
    return parseResponse(result, errorCode);
}

/**
 * Get the authorization token from the Snapchat REST API
 *
 * @param {string} appId The Snapchat OAuth App ID
 * @param {string} appSecret The Snapchat OAuth App Secret
 * @param {string} authCode The Auth code from the Snapchat authentication flow
 * @returns {Object} an object containing the error if any happened
 */
function getAuthToken(appId, appSecret, authCode) {
    var errorCode = 'oauth.access_token.call';
    var service = serviceHelper.getService(constants.SERVICES.SNAPCHAT_ACCOUNTS);
    var params = {
        action: constants.ACTIONS.GET_AUTH_TOKEN,
        method: 'POST',
        path: constants.ENDPOINTS.AUTH,
        params: {
            client_id: appId,
            client_secret: appSecret,
            code: authCode,
            grant_type: 'authorization_code',
            redirect_uri: require('dw/web/URLUtils').https('BM_Snapchat-Callback').toString()
        }
    };
    var result = service.call(params);
    return parseResponse(result, errorCode);
}

/**
 * Get the Snapchat Business Profile
 * @param {dw/object/CustomObject} snapchatSettings The Snapchat settings custom object instance
 * @returns {Object} an object containing the error if any happened
 */
function getBusinessProfile(snapchatSettings) {
    var errorCode = 'get.business.profile.call';
    var service = serviceHelper.getService(constants.SERVICES.SNAPCHAT_ADS);
    var params = {
        action: constants.ACTIONS.GET_BUSINESS_PROFILE,
        method: 'GET',
        path: constants.ENDPOINTS.GET_BUSINESS_PROFILE,
        headers: {
            'Content-Type': constants.CONTENT_TYPE.JSON,
            Authorization: 'Bearer ' + snapchatSettings.custom.accessToken
        }
    };

    var result = service.call(params);
    var response = parseResponse(result, errorCode);
    if (response.error && response.errorCode === 'retry') {
        service = serviceHelper.getService(constants.SERVICES.SNAPCHAT_ADS);
        params.headers.Authorization = 'Bearer ' + snapchatSettings.custom.accessToken;
        result = service.call(params);
        response = parseResponse(result, errorCode);
    }

    return response;
}

/**
 * Get the info for organization
 * @param {dw/object/CustomObject} snapchatSettings The Snapchat settings custom object instance
 * @returns {Object} an object containing the error if any happened
 */
function getOrgInfo(snapchatSettings) {
    var errorCode = 'get.org.info.call';
    var service = serviceHelper.getService(constants.SERVICES.SNAPCHAT_ADS);
    var params = {
        action: constants.ACTIONS.GET_ORG,
        method: 'GET',
        path: StringUtils.format(constants.ENDPOINTS.GET_ORG, snapchatSettings.custom.externalBusinessId),
        params: {
            organization_id: snapchatSettings.custom.externalBusinessId
        },
        headers: {
            'Content-Type': constants.CONTENT_TYPE.JSON,
            Authorization: 'Bearer ' + snapchatSettings.custom.accessToken
        }
    };

    var result = service.call(params);
    var response = parseResponse(result, errorCode);
    if (response.error && response.errorCode === 'retry') {
        service = serviceHelper.getService(constants.SERVICES.SNAPCHAT_ADS);
        params.headers.Authorization = 'Bearer ' + snapchatSettings.custom.accessToken;
        result = service.call(params);
        response = parseResponse(result, errorCode);
    }
    if (!response.error && response.result) {
        if (Object.hasOwnProperty.call(response.result, 'organizations') && !empty(response.result.organizations[0])) {
            // update org info if external business ID was already established during on-boarding.
            if (!empty(snapchatSettings.custom.externalBusinessId) && !empty(response.result.organizations[0].organization.name) && response.result.organizations[0].organization.name !== snapchatSettings.custom.bcId) {
                Transaction.wrap(function () {
                    snapchatSettings.custom.bcId = response.result.organizations[0].organization.name;
                });
            }
        }
    }
    return response;
}

/**
 * Get the list of organizations for a user
 * @param {dw/object/CustomObject} snapchatSettings The Snapchat settings custom object instance
 * @returns {Object} an object containing the error if any happened
 */
function getOrgList(snapchatSettings) {
    var errorCode = 'get.org.list.call';
    var service = serviceHelper.getService(constants.SERVICES.SNAPCHAT_ADS);
    var orgList = new ArrayList();
    var params = {
        action: constants.ACTIONS.GET_ORG_LIST,
        method: 'GET',
        path: constants.ENDPOINTS.GET_ORG_LIST,
        headers: {
            'Content-Type': constants.CONTENT_TYPE.JSON,
            Authorization: 'Bearer ' + snapchatSettings.custom.accessToken
        }
    };

    var result = service.call(params);
    var response = parseResponse(result, errorCode);
    if (response.error && response.errorCode === 'retry') {
        service = serviceHelper.getService(constants.SERVICES.SNAPCHAT_ADS);
        params.headers.Authorization = 'Bearer ' + snapchatSettings.custom.accessToken;
        result = service.call(params);
        response = parseResponse(result, errorCode);
    }
    if (!response.error && response.result) {
        if (Object.hasOwnProperty.call(response.result, 'organizations') && response.result.organizations.length > 0) {
            var orgs = response.result.organizations;
            for (var i = 0; i < orgs.length; i++) {
                var item = orgs[i];
                if (item.organization && item.organization.id && item.organization.state === 'ACTIVE') {
                    var value = [item.organization.id, item.organization.name].filter(Boolean).join('|');
                    var label = [item.organization.name, item.organization.id].filter(Boolean).join(' (');
                    if (label.indexOf('(') > -1) {
                        label += ')';
                    }
                    orgList.push({
                        value: value,
                        label: label
                    });
                }
            }
        }
    }
    orgList.sort(new PropertyComparator('label', true));
    orgList.unshift({
        value: '',
        label: Resource.msg('snapchat.form.select.org', 'snapchat', 'Select')
    });
    response.orgList = orgList;
    return response;
}

/**
 * Get the Ad Accounts for organization
 * @param {dw/object/CustomObject} snapchatSettings The Snapchat settings custom object instance
 * @param {string} orgId The organization id
 * @returns {Object} an object containing the error if any happened
 */
function getAdAccounts(snapchatSettings, orgId) {
    var errorCode = 'get.add.accounts.call';
    var service = serviceHelper.getService(constants.SERVICES.SNAPCHAT_ADS);
    var externalBusinessId = orgId || snapchatSettings.custom.externalBusinessId;
    var adList = new ArrayList();
    var params = {
        action: constants.ACTIONS.GET_AD_ACCOUNTS,
        method: 'GET',
        path: StringUtils.format(constants.ENDPOINTS.GET_AD_ACCOUNTS, externalBusinessId),
        params: {
            organization_id: externalBusinessId
        },
        headers: {
            'Content-Type': constants.CONTENT_TYPE.JSON,
            Authorization: 'Bearer ' + snapchatSettings.custom.accessToken
        }
    };

    var result = service.call(params);
    var response = parseResponse(result, errorCode);
    if (response.error && response.errorCode === 'retry') {
        service = serviceHelper.getService(constants.SERVICES.SNAPCHAT_ADS);
        params.headers.Authorization = 'Bearer ' + snapchatSettings.custom.accessToken;
        result = service.call(params);
        response = parseResponse(result, errorCode);
    }
    if (!response.error && response.result) {
        if (Object.hasOwnProperty.call(response.result, 'adaccounts') && !empty(response.result.adaccounts)) {
            var activeAdAccounts = response.result.adaccounts.filter(function (item) { return item.adaccount.status === 'ACTIVE'; });

            for (var i = 0; i < activeAdAccounts.length; i++) {
                var item = activeAdAccounts[i];
                if (item.adaccount && item.adaccount.id) {
                    var value = [item.adaccount.id, item.adaccount.name].filter(Boolean).join('|');
                    var label = [item.adaccount.name, item.adaccount.id].filter(Boolean).join(' (');
                    if (label.indexOf('(') > -1) {
                        label += ')';
                    }
                    adList.push({
                        value: value,
                        label: label
                    });
                }
            }

            // update ad account if external business ID was already established during on-boarding.
            // ad account was established after on-boarding was completed
            if (activeAdAccounts.length && !empty(snapchatSettings.custom.externalBusinessId)) {
                var activeAdAccount = null;

                if (!empty(snapchatSettings.custom.advertiserId)) {
                    // filter active ad account by id
                    activeAdAccount = activeAdAccounts.filter(function (adAcct) { return adAcct.adaccount.id === snapchatSettings.custom.advertiserId; })[0];
                } else {
                    // set values to first active ad account
                    activeAdAccount = activeAdAccounts[0];
                }

                if (activeAdAccount && activeAdAccount.adaccount && activeAdAccount.adaccount.id) {
                    Transaction.wrap(function () {
                        snapchatSettings.custom.advertiserId = activeAdAccount.adaccount.id;
                        snapchatSettings.custom.advertiserAccount = activeAdAccount.adaccount.name || null;
                    });
                }
            }
        }
    }
    adList.sort(new PropertyComparator('label', true));
    adList.unshift({
        value: '',
        label: Resource.msg('snapchat.form.select.ad.account', 'snapchat', 'Select')
    });

    response.adList = adList;
    return response;
}

/**
 * Get the PixelId for the Ad Account
 * @param {dw/object/CustomObject} snapchatSettings The Snapchat settings custom object instance
 * @param {string} adAccountId The Ad Account ID
 * @returns {Object} an object containing the error if any happened
 */
function getPixelFromAdAccount(snapchatSettings, adAccountId) {
    var errorCode = 'get.pixel.adaccount.call';
    var service = serviceHelper.getService(constants.SERVICES.SNAPCHAT_ADS);
    var advertiserId = adAccountId || snapchatSettings.custom.advertiserId;
    var pixelList = new ArrayList();
    var params = {
        action: constants.ACTIONS.GET_PIXEL_FROM_AD_ACCOUNT,
        method: 'GET',
        path: StringUtils.format(constants.ENDPOINTS.GET_PIXEL_FROM_AD_ACCOUNT, advertiserId),
        params: {
            ad_account_id: advertiserId
        },
        headers: {
            'Content-Type': constants.CONTENT_TYPE.JSON,
            Authorization: 'Bearer ' + snapchatSettings.custom.accessToken
        }
    };

    var result = service.call(params);
    var response = parseResponse(result, errorCode);
    if (response.error && response.errorCode === 'retry') {
        service = serviceHelper.getService(constants.SERVICES.SNAPCHAT_ADS);
        params.headers.Authorization = 'Bearer ' + snapchatSettings.custom.accessToken;
        result = service.call(params);
        response = parseResponse(result, errorCode);
    }
    if (!response.error && response.result) {
        if (Object.hasOwnProperty.call(response.result, 'pixels') && !empty(response.result.pixels)) {
            var activePixels = response.result.pixels.filter(function (item) { return item.pixel.status === 'ACTIVE'; });

            for (var i = 0; i < activePixels.length; i++) {
                var item = activePixels[i];
                if (item.pixel && item.pixel.id) {
                    var value = [item.pixel.id, item.pixel.name].filter(Boolean).join('|');
                    var label = [item.pixel.name, item.pixel.id].filter(Boolean).join(' (');
                    if (label.indexOf('(') > -1) {
                        label += ')';
                    }
                    pixelList.push({
                        value: value,
                        label: label
                    });
                }
            }

            // update pixel if external business ID was already established during on-boarding.
            // pixel was established after on-boarding was completed
            if (activePixels.length && !empty(snapchatSettings.custom.externalBusinessId)) {
                var activePixel = null;
                var externalData = customObjectHelper.getExternalData(snapchatSettings) || {};

                if (!empty(snapchatSettings.custom.pixelCode)) {
                    // filter active pixel by id
                    activePixel = activePixels.filter(function (pixelItem) { return pixelItem.pixel.id === snapchatSettings.custom.pixelCode; })[0];
                } else {
                    // set values to first active pixel
                    activePixel = activePixels[0];
                }

                if (activePixel && activePixel.pixel && activePixel.pixel.id) {
                    Transaction.wrap(function () {
                        snapchatSettings.custom.pixelCode = activePixel.pixel.id;
                        if (activePixel.pixel.name) {
                            externalData.pixelName = activePixel.pixel.name;
                            snapchatSettings.custom.externalData = JSON.stringify(externalData);
                        }
                    });
                }
            }
        }
    }
    pixelList.sort(new PropertyComparator('label', true));
    pixelList.unshift({
        value: '',
        label: Resource.msg('snapchat.form.select.pixel', 'snapchat', 'Select')
    });
    response.pixelList = pixelList;
    return response;
}

/**
 * Disconnect from Snapchat
 * @param {dw/object/CustomObject} snapchatSettings The Snapchat settings custom object instance
 * @returns {boolean} True if the disconnect process succeed, false otherwise
 */
function disconnectFromSnapchat(snapchatSettings) {
    var service = serviceHelper.getService(constants.SERVICES.SNAPCHAT_ACCOUNTS); // client_id=CLIENT_ID&token=REFRESH_TOKEN
    var params = {
        action: constants.ACTIONS.DISCONNECT,
        method: 'POST',
        path: constants.ENDPOINTS.DISCONNECT + '?client_id=' + snapchatSettings.custom.appId + '&token=' + snapchatSettings.custom.refreshToken,
        headers: {
            'Content-Type': constants.CONTENT_TYPE.URLENCODED,
            Authorization: 'Basic ' + StringUtils.encodeBase64(snapchatSettings.custom.appId + ':' + snapchatSettings.custom.appSecret)
        }
    };
    var result = service.call(params);
    if (result.error) {
        Logger.error('Error occurred while disconnecting from Snapchat: ' + result.error);
        return false;
    }
    return true;
}

/**
 * Get product feed ID from the service response.
 * @param {Object} parsedResponse The response from the service.
 * @returns {string} The product feed ID.
 */
function getProductFeedIdFromSvcResponse(parsedResponse) {
    return !empty(parsedResponse) && !empty(parsedResponse.result)
    && Object.hasOwnProperty.call(parsedResponse.result, 'product_feeds')
    && !empty(parsedResponse.result.product_feeds[0])
    && Object.hasOwnProperty.call(parsedResponse.result.product_feeds[0], 'product_feed')
    && Object.hasOwnProperty.call(parsedResponse.result.product_feeds[0].product_feed, 'id')
    && !empty(parsedResponse.result.product_feeds[0].product_feed.id) ? parsedResponse.result.product_feeds[0].product_feed.id : null;
}

/**
 * Create the product feed
 * @param {dw/object/CustomObject} snapchatSettings The Snapchat settings custom object instance
 * @param {string} catalogId The Snapchat catalog ID
 * @param {string} hostName The instance hostname
 * @returns {Object} an object containing the error if any happened
 */
function createProductFeed(snapchatSettings, catalogId, hostName) {
    var errorCode = 'create.product.feed.call';
    var service = serviceHelper.getService(constants.SERVICES.SNAPCHAT_ADS);
    var siteId = Site.getCurrent().ID;
    var externalData = customObjectHelper.getExternalData(snapchatSettings);

    var params = {
        action: constants.ACTIONS.CREATE_PRODUCT_FEED,
        method: 'POST',
        path: StringUtils.format(constants.ENDPOINTS.CREATE_PRODUCT_FEED, catalogId),
        headers: {
            'Content-Type': constants.CONTENT_TYPE.JSON,
            Authorization: 'Bearer ' + snapchatSettings.custom.accessToken
        },
        body: {
            product_feeds: [{
                name: externalData && externalData.productFeedName ? externalData.productFeedName : 'Salesforce Product Feed (' + siteId + ')',
                default_currency: externalData && externalData.catalogCurrencyCode ? externalData.catalogCurrencyCode : Site.getCurrent().getDefaultCurrency(),
                status: constants.FEEDS.PRODUCT.STATUS,
                source: constants.FEEDS.SOURCE,
                partner_platform_connection_data: {
                    salesforce_connection_data: {
                        short_code: externalData && externalData.shortCode ? externalData.shortCode : '',
                        http_hostname: 'https://' + hostName + '/',
                        auth_credentials: {
                            client_id: snapchatSettings.custom.shopperClientId,
                            client_secret: snapchatSettings.custom.shopperClientSecret
                        }
                    }
                }
            }]
        }
    };

    var result = service.call(params);
    var parsedResponse = parseResponse(result, errorCode);

    if (parsedResponse.error || parsedResponse.result.request_status === 'ERROR') {
        errorCode = !empty(parsedResponse.result.display_message)
            ? Resource.msgf('snapchat.error.create.product.feed.svcError', 'snapchat', null, parsedResponse.result.display_message)
            : errorCode;
        return {
            error: errorCode
        };
    }

    var productFeedId = getProductFeedIdFromSvcResponse(parsedResponse);
    if (!productFeedId) {
        return {
            error: errorCode
        };
    }

    // save the product feed ID
    Transaction.wrap(function () {
        snapchatSettings.custom.productFeedId = productFeedId;
    });

    return parsedResponse;
}

/**
 * Get the catalog ID from the service response.
 * @param {Object} parsedResponse The response from the service.
 * @returns {string} The catalog ID.
 */
function getCatalogIdFromSvcResponse(parsedResponse) {
    return !empty(parsedResponse) && !empty(parsedResponse.result)
        && Object.hasOwnProperty.call(parsedResponse.result, 'catalogs')
        && !empty(parsedResponse.result.catalogs[0])
        && Object.hasOwnProperty.call(parsedResponse.result.catalogs[0], 'catalog')
        && Object.hasOwnProperty.call(parsedResponse.result.catalogs[0].catalog, 'id')
        && !empty(parsedResponse.result.catalogs[0].catalog.id) ? parsedResponse.result.catalogs[0].catalog.id : null;
}

/**
 * build external store ID
 * @param {dw/object/CustomObject} snapchatSettings The Snapchat settings custom object instance
 * @returns {string} external store ID (f_ecom_zzte_053|RefArch)
 */
function getExternalStoreId(snapchatSettings) {
    var hostName = System.getInstanceHostname();
    var externalData = customObjectHelper.getExternalData(snapchatSettings);
    var orgId = externalData && externalData.orgId ? externalData.orgId : null;
    var siteId = Site.getCurrent().ID;
    if (!orgId && hostName.indexOf('dx.commercecloud') > -1) {
        orgId = 'f_ecom_' + hostName.split('.').shift().replace('-', '_');
    }
    return [orgId, siteId].filter(Boolean).join('|');
}

/**
 * Create the catalog feed
 * @param {dw/object/CustomObject} snapchatSettings The Snapchat settings custom object instance
 * @returns {Object} an object containing the error if any happened
 */
function createCatalogFeed(snapchatSettings) {
    var service;
    var svcResult;
    var svcParams;
    var parsedResponse;
    var catalogId = null;
    var hostName = System.getInstanceHostname();
    var siteId = Site.getCurrent().ID;
    var errorCode = 'create.catalog.call';

    // make sure catalog ID doesn't already exist in Snapchat
    if (!empty(snapchatSettings.custom.catalogId)) {
        service = serviceHelper.getService(constants.SERVICES.SNAPCHAT_ADS);
        svcParams = {
            action: constants.ACTIONS.GET_CATALOG,
            method: 'GET',
            path: StringUtils.format(constants.ENDPOINTS.GET_CATALOG, snapchatSettings.custom.catalogId),
            headers: {
                'Content-Type': constants.CONTENT_TYPE.JSON,
                Authorization: 'Bearer ' + snapchatSettings.custom.accessToken
            }
        };
        svcResult = service.call(svcParams);
        parsedResponse = parseResponse(svcResult, errorCode);
        if (getCatalogIdFromSvcResponse(parsedResponse) === snapchatSettings.custom.catalogId) {
            catalogId = snapchatSettings.custom.catalogId;
        }
    }

    if (!catalogId) {
        var externalData = customObjectHelper.getExternalData(snapchatSettings);
        var catalogName = externalData && externalData.catalogName ? externalData.catalogName : 'Salesforce Catalog (' + siteId + ')';
        svcParams = {
            action: constants.ACTIONS.CREATE_CATALOG,
            method: 'POST',
            path: StringUtils.format(constants.ENDPOINTS.GET_ORG_CATALOGS, snapchatSettings.custom.externalBusinessId),
            headers: {
                'Content-Type': constants.CONTENT_TYPE.JSON,
                Authorization: 'Bearer ' + snapchatSettings.custom.accessToken
            },
            body: {
                catalogs: [
                    {
                        organization_id: snapchatSettings.custom.externalBusinessId,
                        name: catalogName,
                        vertical: constants.FEEDS.CATALOG.VERTICAL,
                        source: constants.FEEDS.SOURCE,
                        external_store_id: getExternalStoreId(snapchatSettings),
                        settings: {
                            commerce_settings: {
                                native_checkout_eligibility: constants.FEEDS.CATALOG.NATIVE_CHECKOUT_ELIGIBILITY,
                                merchant_platform_metadata: {
                                    salesforce_connection_data: {
                                        http_hostname: 'https://' + hostName
                                    }
                                }
                            }
                        }
                    }]
            }
        };
        service = serviceHelper.getService(constants.SERVICES.SNAPCHAT_ADS);
        svcResult = service.call(svcParams);
        parsedResponse = parseResponse(svcResult, errorCode);

        if (parsedResponse.error || parsedResponse.result.request_status === 'ERROR') {
            errorCode = !empty(parsedResponse.result.catalogs[0])
            && !empty(parsedResponse.result.catalogs[0].sub_request_error_reason)
                ? Resource.msgf('snapchat.error.create.catalog.svcError', 'snapchat', null, parsedResponse.result.catalogs[0].sub_request_error_reason)
                : errorCode;
            return {
                error: errorCode
            };
        }

        catalogId = getCatalogIdFromSvcResponse(parsedResponse);
        if (catalogId) {
            // save the catalog ID
            Transaction.wrap(function () {
                snapchatSettings.custom.catalogId = catalogId;
            });
        }
    }

    if (!catalogId) {
        return {
            error: errorCode
        };
    }

    // create product feed
    return createProductFeed(snapchatSettings, catalogId, hostName);
}

/**
 * Get the catalogs for this organization
 * @param {dw/object/CustomObject} snapchatSettings The Snapchat settings custom object instance
 * @param {string} orgId The organization ID
 * @param {Array} catalogs current list of catalogs
 * @param {Object} reqParams optional request params
 * @returns {Object} an object containing the error if any happened
 */
function getCatalogs(snapchatSettings, orgId, catalogs, reqParams) {
    var errorCode = 'get.catalogs.call';
    var service = serviceHelper.getService(constants.SERVICES.SNAPCHAT_ADS);
    var externalBusinessId = orgId || snapchatSettings.custom.externalBusinessId;
    var activeCatalogs = catalogs || [];
    var catalogList = new ArrayList();
    var svcParams = {
        action: constants.ACTIONS.GET_ORG_CATALOGS,
        method: 'GET',
        path: StringUtils.format(constants.ENDPOINTS.GET_ORG_CATALOGS, externalBusinessId),
        params: reqParams || { limit: constants.LIMITS.GET_ORG_CATALOGS },
        headers: {
            'Content-Type': constants.CONTENT_TYPE.JSON,
            Authorization: 'Bearer ' + snapchatSettings.custom.accessToken
        }
    };
    var svcResult = service.call(svcParams);
    var parsedResponse = parseResponse(svcResult, errorCode);
    if (parsedResponse.error && parsedResponse.errorCode === 'retry') {
        service = serviceHelper.getService(constants.SERVICES.SNAPCHAT_ADS);
        svcParams.headers.Authorization = 'Bearer ' + snapchatSettings.custom.accessToken;
        svcResult = service.call(svcParams);
        parsedResponse = parseResponse(svcResult, errorCode);
    }

    if (!parsedResponse.error && parsedResponse.result) {
        if (Object.hasOwnProperty.call(parsedResponse.result, 'catalogs') && parsedResponse.result.catalogs.length) {
            activeCatalogs = activeCatalogs.concat(parsedResponse.result.catalogs.filter(function (item) { return item.catalog.vertical === constants.FEEDS.CATALOG.VERTICAL && item.catalog.source === constants.FEEDS.SOURCE; }));
        }

        if (Object.hasOwnProperty.call(parsedResponse.result, 'paging')) {
            var nextLinkParams = { };
            if (Object.hasOwnProperty.call(parsedResponse.result.paging, 'next_link') && parsedResponse.result.paging.next_link) {
                var urlParts = parsedResponse.result.paging.next_link.split('?');
                var queryString = urlParts.length > 0 ? urlParts[1] : '';
                if (queryString) {
                    nextLinkParams = serviceHelper.parseQueryString(queryString);
                }
            }
            if (!nextLinkParams.cursor && Object.hasOwnProperty.call(parsedResponse.result.paging, 'next_cursor') && parsedResponse.result.paging.next_cursor) {
                nextLinkParams.cursor = parsedResponse.result.paging.next_cursor;
            }
            if (nextLinkParams.cursor) {
                if (!nextLinkParams.limit) {
                    nextLinkParams.limit = constants.LIMITS.GET_ORG_CATALOGS;
                }
                getCatalogs(snapchatSettings, externalBusinessId, activeCatalogs, nextLinkParams);
            }
        }
    }

    if (activeCatalogs.length) {
        for (var i = 0; i < activeCatalogs.length; i++) {
            var item = activeCatalogs[i];
            if (item.catalog && item.catalog.id) {
                var value = [item.catalog.id, item.catalog.name].filter(Boolean).join('|');
                var label = [item.catalog.name, item.catalog.id].filter(Boolean).join(' (');
                if (label.indexOf('(') > -1) {
                    label += ')';
                }
                catalogList.push({
                    value: value,
                    label: label
                });
            }
        }
        // update catalog if external business ID was already established during on-boarding.
        // catalog was established after on-boarding was completed
        if (!empty(snapchatSettings.custom.externalBusinessId)) {
            var activeCatalog = null;
            var externalData = customObjectHelper.getExternalData(snapchatSettings) || {};

            if (!empty(snapchatSettings.custom.catalogId)) {
                // filter active catalog by id
                activeCatalog = activeCatalogs.filter(function (catalogItem) { return catalogItem.catalog.id === snapchatSettings.custom.catalogId; })[0];
            } else {
                // set values to first active catalog
                activeCatalog = activeCatalogs[0];
            }

            if (activeCatalog && activeCatalog.catalog && activeCatalog.catalog.id) {
                Transaction.wrap(function () {
                    snapchatSettings.custom.catalogId = activeCatalog.catalog.id;
                    if (activeCatalog.catalog.name) {
                        externalData.catalogName = activeCatalog.catalog.name;
                        snapchatSettings.custom.externalData = JSON.stringify(externalData);
                    }
                });
            }
        }
    } else if (!empty(snapchatSettings.custom.externalBusinessId) && !empty(snapchatSettings.custom.catalogId)) {
        // check to see if catalog still exists, if not clear catalog from settings
        service = serviceHelper.getService(constants.SERVICES.SNAPCHAT_ADS);
        svcParams = {
            action: constants.ACTIONS.GET_CATALOG,
            method: 'GET',
            path: StringUtils.format(constants.ENDPOINTS.GET_CATALOG, snapchatSettings.custom.catalogId),
            headers: {
                'Content-Type': constants.CONTENT_TYPE.JSON,
                Authorization: 'Bearer ' + snapchatSettings.custom.accessToken
            }
        };
        svcResult = service.call(svcParams);
        parsedResponse = parseResponse(svcResult, errorCode);
        if (getCatalogIdFromSvcResponse(parsedResponse) !== snapchatSettings.custom.catalogId) {
            // clear catalog id if it no longer exists in Snapchat
            Transaction.wrap(function () {
                snapchatSettings.custom.catalogId = null;
                snapchatSettings.custom.productFeedId = null;
                snapchatSettings.custom.feedUploadId = null;
            });
        }
    }

    catalogList.sort(new PropertyComparator('label', true));
    catalogList.unshift({
        value: '',
        label: Resource.msg('snapchat.form.select.catalog', 'snapchat', 'Select')
    });
    parsedResponse.catalogList = catalogList;
    return parsedResponse;
}

/**
 * Get the product feed for the catalog ID
 * @param {dw/object/CustomObject} snapchatSettings The Snapchat settings custom object instance
 * @param {string} catId The catalog ID
 * @returns {Object} an object containing the error if any happened
 */
function getProductFeeds(snapchatSettings, catId) {
    var errorCode = 'get.product.feeds.call';
    var service = serviceHelper.getService(constants.SERVICES.SNAPCHAT_ADS);
    var catalogId = catId || snapchatSettings.custom.catalogId;
    var productFeedList = new ArrayList();
    var params = {
        action: constants.ACTIONS.GET_PRODUCT_FEEDS,
        method: 'GET',
        path: StringUtils.format(constants.ENDPOINTS.CREATE_PRODUCT_FEED, catalogId),
        headers: {
            'Content-Type': constants.CONTENT_TYPE.JSON,
            Authorization: 'Bearer ' + snapchatSettings.custom.accessToken
        }
    };

    var svcResult = service.call(params);
    var parsedResponse = parseResponse(svcResult, errorCode);
    if (parsedResponse.error && parsedResponse.errorCode === 'retry') {
        service = serviceHelper.getService(constants.SERVICES.SNAPCHAT_ADS);
        params.headers.Authorization = 'Bearer ' + snapchatSettings.custom.accessToken;
        svcResult = service.call(params);
        parsedResponse = parseResponse(svcResult, errorCode);
    }
    if (!parsedResponse.error && parsedResponse.result) {
        if (Object.hasOwnProperty.call(parsedResponse.result, 'product_feeds')) {
            if (parsedResponse.result.product_feeds.length === 0) {
                // clear product feed id if it no longer exists in Snapchat
                if (!empty(snapchatSettings.custom.externalBusinessId) && !empty(snapchatSettings.custom.productFeedId)) {
                    Transaction.wrap(function () {
                        snapchatSettings.custom.productFeedId = null;
                        snapchatSettings.custom.feedUploadId = null;
                    });
                }
            } else {
                var activeProductFeeds = parsedResponse.result.product_feeds.filter(function (item) { return item.product_feed.source === constants.FEEDS.SOURCE; });

                for (var i = 0; i < activeProductFeeds.length; i++) {
                    var item = activeProductFeeds[i];
                    if (item.product_feed && item.product_feed.id) {
                        var value = [item.product_feed.id, item.product_feed.name].filter(Boolean).join('|');
                        var label = [item.product_feed.name, item.product_feed.id].filter(Boolean).join(' (');
                        if (label.indexOf('(') > -1) {
                            label += ')';
                        }
                        productFeedList.push({
                            value: value,
                            label: label
                        });
                    }
                }

                // update product feed if external business ID was already established during on-boarding.
                // product feed was established after on-boarding was completed
                if (activeProductFeeds.length && !empty(snapchatSettings.custom.externalBusinessId)) {
                    var activeProductFeed = null;
                    var externalData = customObjectHelper.getExternalData(snapchatSettings) || {};

                    if (!empty(snapchatSettings.custom.productFeedId)) {
                        // filter active product feed by id
                        activeProductFeed = activeProductFeeds.filter(function (productFeedItem) { return productFeedItem.product_feed.id === snapchatSettings.custom.productFeedId; })[0];
                    } else {
                        // set values to first active product feed
                        activeProductFeed = activeProductFeeds[0];
                    }

                    if (activeProductFeed && activeProductFeed.product_feed && activeProductFeed.product_feed.id) {
                        Transaction.wrap(function () {
                            snapchatSettings.custom.productFeedId = activeProductFeed.product_feed.id;
                            if (activeProductFeed.product_feed.name) {
                                externalData.productFeedName = activeProductFeed.product_feed.name;
                                snapchatSettings.custom.externalData = JSON.stringify(externalData);
                            }
                        });
                    }
                }
            }
        }
    }
    productFeedList.sort(new PropertyComparator('label', true));
    productFeedList.unshift({
        value: '',
        label: Resource.msg('snapchat.form.select.productFeed', 'snapchat', 'Select')
    });
    parsedResponse.productFeedList = productFeedList;
    return parsedResponse;
}

/**
 * Create Product Feed Upload/notify
 * @param {dw/object/CustomObject} snapchatSettings The Snapchat settings custom object instance
 * @param {string} url  the full path to the product file
 * @param {string} updateType - REPLACE or UPSERT
 * @returns {Object} an object containing the error if any happened
 */
function createFeedUpload(snapchatSettings, url, updateType) {
    var errorCode = 'create.feed.upload.call';
    var service = serviceHelper.getService(constants.SERVICES.SNAPCHAT_ADS);
    var productFeedId = snapchatSettings.custom.productFeedId;
    if (!productFeedId) {
        return {
            error: true,
            errorCode: 'NO_PRODUCT_FEED_ID'
        };
    }
    var params = {
        action: constants.ACTIONS.CREATE_FEED_UPLOAD,
        method: 'POST',
        path: StringUtils.format(constants.ENDPOINTS.CREATE_FEED_UPLOAD, productFeedId),
        headers: {
            'Content-Type': constants.CONTENT_TYPE.JSON,
            Authorization: 'Bearer ' + snapchatSettings.custom.accessToken
        },
        body: {
            feed_uploads: [{
                url: url,
                username: snapchatSettings.custom.bizMngrUsername,
                password: snapchatSettings.custom.bizMngrAccessKeyWebdav,
                update_type: updateType || 'REPLACE'
            }]
        }
    };

    var result = service.call(params);
    var response = parseResponse(result, errorCode);
    if (response.error && response.errorCode === 'retry') {
        service = serviceHelper.getService(constants.SERVICES.SNAPCHAT_ADS);
        params.headers.Authorization = 'Bearer ' + snapchatSettings.custom.accessToken;
        result = service.call(params);
        response = parseResponse(result, errorCode);
    }
    if (!response.error && response.result && Object.hasOwnProperty.call(response.result, 'feed_uploads') && !empty(response.result.feed_uploads)) {
        var feedUploads = response.result.feed_uploads[0];
        if (Object.hasOwnProperty.call(feedUploads, 'feed_upload') && Object.hasOwnProperty.call(feedUploads.feed_upload, 'id')) {
            response.feedUploadId = feedUploads.feed_upload.id;
            Transaction.wrap(function () {
                snapchatSettings.custom.feedUploadId = feedUploads.feed_upload.id;
            });
        }
    }
    return response;
}

/**
 * Get Product Feed Upload
 * @param {dw/object/CustomObject} snapchatSettings The Snapchat settings custom object instance
 * @param {string} feedId - the feed upload id
 * @returns {Object} an object containing the error if any happened
 */
function getFeedUpload(snapchatSettings, feedId) {
    var errorCode = 'get.feed.upload.call';
    var service = serviceHelper.getService(constants.SERVICES.SNAPCHAT_ADS);
    var feedUploadId = feedId || snapchatSettings.custom.feedUploadId;
    if (!feedUploadId) {
        return {
            error: true,
            errorCode: 'NO_FEED_UPLOAD_ID'
        };
    }
    var params = {
        action: constants.ACTIONS.GET_FEED_UPLOAD,
        method: 'GET',
        path: StringUtils.format(constants.ENDPOINTS.GET_FEED_UPLOAD, feedUploadId),
        headers: {
            'Content-Type': constants.CONTENT_TYPE.JSON,
            Authorization: 'Bearer ' + snapchatSettings.custom.accessToken
        }
    };

    var result = service.call(params);
    var response = parseResponse(result, errorCode);
    if (response.error && response.errorCode === 'retry') {
        service = serviceHelper.getService(constants.SERVICES.SNAPCHAT_ADS);
        params.headers.Authorization = 'Bearer ' + snapchatSettings.custom.accessToken;
        result = service.call(params);
        response = parseResponse(result, errorCode);
    }
    if (!response.error && response.result && Object.hasOwnProperty.call(response.result, 'feed_uploads') && !empty(response.result.feed_uploads)) {
        var feedUploads = response.result.feed_uploads[0];
        if (Object.hasOwnProperty.call(feedUploads, 'feed_upload')) {
            response.feedUpload = feedUploads.feed_upload;
            if (Object.hasOwnProperty.call(feedUploads.feed_upload, 'status') && feedUploads.feed_upload.status) {
                response.feedUploadStatus = feedUploads.feed_upload.status;
            }
        }
    }
    return response;
}

/**
 * send feed notification
 *
 * @param {dw.object.CustomObject} snapchatSettings The SnapChat settings custom object instance
 * @param {dw/object/CustomObject} instance The instance custom object instance
 * @param {string} feedURL The feed URL
 * @param {string} feedType The feed type
 * @param {string} updateType The update type
 * @returns {boolean} True if the upload process succeed, false otherwise
 */
function notifyFeed(snapchatSettings, instance, feedURL, feedType, updateType) {
    var errorCode = 'notify.order.call';
    var service = serviceHelper.getService(constants.SERVICES.SNAPCHAT_NOTIFY);
    var params = {
        action: constants.ACTIONS.NOTIFY_ORDER,
        method: 'POST',
        path: constants.ENDPOINTS.NOTIFY_ORDER,
        headers: {
            'Content-Type': constants.CONTENT_TYPE.JSON,
            Authorization: 'Bearer ' + snapchatSettings.custom.accessToken
        },
        body: {
            business_platform: constants.BUSINESS_PLATFORM,
            external_business_id: getExternalStoreId(snapchatSettings),
            params: {
                url: feedURL,
                feed_type: feedType,
                update_type: updateType
            }
        }
    };

    var result = service.call(params);
    var svcResponse = parseResponse(result, errorCode);
    if (svcResponse.error && svcResponse.errorCode === 'retry') {
        service = serviceHelper.getService(constants.SERVICES.SNAPCHAT_NOTIFY);
        params.headers.Authorization = 'Bearer ' + snapchatSettings.custom.accessToken;
        result = service.call(params);
    }
    if (result.ok) {
        return true;
    }
    Logger.error('Error occurred calling Snapchat order notify service: ' + result.errorMessage || '');
    return false;
}

module.exports = {
    createCatalogFeed: createCatalogFeed,
    disconnectFromSnapchat: disconnectFromSnapchat,
    getAdAccounts: getAdAccounts,
    getAuthToken: getAuthToken,
    getBusinessProfile: getBusinessProfile,
    getCatalogs: getCatalogs,
    getOrgInfo: getOrgInfo,
    getOrgList: getOrgList,
    getPixelFromAdAccount: getPixelFromAdAccount,
    getProductFeeds: getProductFeeds,
    createFeedUpload: createFeedUpload,
    getFeedUpload: getFeedUpload,
    notifyFeed: notifyFeed
};
