'use strict';

var File = require('dw/io/File');
var FileReader = require('dw/io/FileReader');
var Logger = require('dw/system/Logger').getLogger('Snapchat');
var Result = require('dw/svc/Result');
var Status = require('dw/system/Status');
var StringUtils = require('dw/util/StringUtils');
var System = require('dw/system/System');

var constants = require('~/cartridge/scripts/SnapchatConstants');
var customObjectHelper = require('~/cartridge/scripts/customObjectHelper');
var fileHelper = require('int_socialfeeds/cartridge/scripts/helpers/FileHelper.js');
var snapchatService = require('~/cartridge/scripts/services/snapchatService');

var MAX_RETRY_ATTEMPTS = 3;
var MAX_STATUS_ATTEMPTS = 50;

/**
 * ensures file exists and has contents
 * @param {dw.io.File} file file to process
 * @returns {boolean} true/false
 */
function checkFileContents(file) {
    if (!file || !file.exists()) {
        return false;
    }

    // if file is empty, delete it do not post to Snapchat
    var reader = new FileReader(file);
    var deleteFile = false;
    var lineCount = 0;
    while ((reader.readLine()) != null && lineCount < 2) {
        lineCount++;
    }
    reader.close();
    if (lineCount <= 1) {
        deleteFile = true;
    }

    if (!deleteFile) return true;

    try {
        file.remove();
        Logger.debug('file was empty, deleted temp file {0}', file.name);
    } catch (e) {
        Logger.error(e.toString() + ' in ' + e.fileName + ':' + e.lineNumber);
    }

    return false;
}

/**
 * Returns the update type of the file.
 * @param {string} fileName - the file name
 * @returns {null|string} null or the update type of the file
 */
function getUpdateType(fileName) {
    var baseRegExp = '^{{updateType}}-.*\\.(csv|txt|xml)$';
    var deltaRegExp = new RegExp(baseRegExp.replace('{{updateType}}', 'delta'), 'gi');
    var fullRegExp = new RegExp(baseRegExp.replace('{{updateType}}', 'full'), 'gi');
    var deleteRegExp = new RegExp(baseRegExp.replace('{{updateType}}', 'delete'), 'gi');

    if (deltaRegExp.test(fileName)) {
        return 'UPSERT';
    }
    if (fullRegExp.test(fileName)) {
        return 'REPLACE';
    }
    // delete files are not supported for Snapchat
    if (deleteRegExp.test(fileName)) {
        return null;
    }
    // default to REPLACE update type if unknown
    return 'REPLACE';
}

/**
 * Sleeps for the specified number of milliseconds.
 * WARNING: This function should be used with care and
 * only when absolutely necessary to prevent thread pool exhaustion.
 *
 * the web service call to an invalid port should block on the request
 * and hopefully this block should prevent the CPU from running the process
 *
 * per Class HTTPClient.setTimeout documentation:
 * This timeout value controls both the "connection timeout"
 * (how long it takes to connect to the remote host)
 * and the "socket timeout" (how long, after connecting, it will wait without any data being read).
 * Therefore, in the worst case scenario, the total time of inactivity
 * could be twice as long as the specified value.
 * Because of this, the timeout is divided by 2 to ensure that the
 * total time we are sleeping is not longer than the specified value.
 * @param {number} ms - millisecond to wait
 */
function sleep(ms) {
    try {
        var HTTPClient = require('dw/net/HTTPClient');
        var httpClient = new HTTPClient();
        httpClient.setTimeout(ms / 2);
        httpClient.open('GET', 'https://' + System.getInstanceHostname() + ':9999/');
        httpClient.send();
    } catch (e) { /* ignore */ }
}

/**
 * Checks if the given service result is an error.
 * @param {Object} svcResult - the service result
 * @returns {boolean} true if the given service result is an error, false otherwise
 */
function isSvcReqError(svcResult) {
    return !svcResult.error
        && Object.hasOwnProperty.call(svcResult, 'result')
        && Object.hasOwnProperty.call(svcResult.result, 'request_status')
        && svcResult.result.request_status === 'ERROR';
}

/**
 * Calls Snapchat service to creates a feed upload.
 * @param {Object} snapchatSettings - the snapchat settings
 * @param {string} feedUploadUrl - the url to POST the feed upload to
 * @param {string} updateType - the update type of the feed upload
 * @returns {null|string} the feed upload id
 */
function createFeedUpload(snapchatSettings, feedUploadUrl, updateType) {
    if (!feedUploadUrl || !updateType) {
        return null;
    }

    var attempts = 0;

    while (attempts < MAX_RETRY_ATTEMPTS) {
        var createResult = snapchatService.createFeedUpload(snapchatSettings, feedUploadUrl, updateType);
        attempts++;

        // if there was a problem with the request, exit early
        if (isSvcReqError(createResult)) {
            return null;
        }

        // if the feed upload ID exists, return it
        if (!createResult.error && createResult.feedUploadId) {
            return createResult.feedUploadId;
        }

        // wait before attempting the next service call
        if (createResult.errorCode === Result.SERVICE_UNAVAILABLE) {
            sleep(3000);
        }
    }

    return null;
}

/**
 * Checks the status of a feed upload.
 * @param {Object} snapchatSettings - the snapchat settings
 * @param {string} feedUploadId - the id of the feed upload
 * @param {dw.io.File} baseDir - the base path to the impex product directory
 * @param {dw.io.File} processedDir - the directory to store the processed files
 * @returns {dw.system.Status} true if the feed upload is complete, false otherwise
 */
function checkFeedUploadStatus(snapchatSettings, feedUploadId, baseDir, processedDir) {
    if (!feedUploadId) return false;

    var attempts = 0;
    var COMPLETE_STATUS = [
        constants.FEED_UPLOAD_STATUS.COMPLETE,
        constants.FEED_UPLOAD_STATUS.ERRORED
    ];
    var errorMessage;

    while (attempts < MAX_STATUS_ATTEMPTS) {
        var statusResult = snapchatService.getFeedUpload(snapchatSettings, feedUploadId);
        attempts++;

        // if there was a problem with the request, exit early
        if (isSvcReqError(statusResult)) {
            errorMessage = 'There was an error with the feed upload status request. Please check the log for more details.';
            Logger.error(errorMessage);
            return new Status(Status.ERROR, 'ERROR', errorMessage);
        }

        // move file to processed folder and return true if the feed upload is finished processing
        var feedUpload = !statusResult.error && Object.hasOwnProperty.call(statusResult, 'feedUpload') ? statusResult.feedUpload : null;
        var feedUploadStatus = feedUpload && Object.hasOwnProperty.call(feedUpload, 'status') ? feedUpload.status : null;
        var feedUrl = feedUpload && Object.hasOwnProperty.call(feedUpload, 'url') ? feedUpload.url : null;
        if (feedUploadStatus && COMPLETE_STATUS.indexOf(feedUploadStatus) > -1) {
            // move file to processed folder
            if (baseDir.exists() && processedDir.exists() && feedUrl) {
                var fileName = feedUrl.split('/').pop();
                var file = new File(baseDir, fileName);
                if (file.exists()) {
                    fileHelper.moveFile(file, processedDir);
                }
            }
            return new Status(Status.OK);
        }

        // wait before attempting the next service call
        sleep(5000);
    }

    errorMessage = StringUtils.format('Could not get a successful feed upload status after {0} attempts.  Please check the log for more details.', MAX_STATUS_ATTEMPTS);
    Logger.warn(errorMessage);
    return new Status(Status.ERROR, 'RETRY_EXCEEDED', errorMessage);
}

/**
 * Returns the file pattern for the given parameters.
 * @param {Object} params - the parameters
 * @returns {string} the file pattern
 */
function getFilePattern(params) {
    var Site = require('dw/system/Site');

    if (!params.FilePattern) return null;

    return params.FilePattern.replace(/\{\{[^}]*\}\}/g, function (a) {
        var parts = a.split(/(?:\{\{| |\}\})/g);
        var variable = parts[1];
        if (!variable) return '';

        switch (variable) {
            case 'site_id':
            case 'siteId':
                return Site.getCurrent().ID;
            case 'siteName':
            case 'site_name':
                return Site.getCurrent().getName();
            default:
                return '';
        }
    });
}

/**
 * Creates Snapchat feed upload API call based on latest catalog export
 * @param {Object} params job parameters
 * @returns {dw.system.Status} Exit status for a job run
 */
exports.execute = function (params) {
    var snapchatSettings = customObjectHelper.getCustomObject();
    var impexPath = params.FileFolder || constants.FEED_PATHS.PRODUCT;
    var filePattern = getFilePattern(params);
    var baseDirPath = File.IMPEX + impexPath;
    var webDavFilePath = StringUtils.format(
        '{0}://{1}{2}{3}',
        'https',
        System.getInstanceHostname(),
        constants.IMPEX_DEFAULT_PATH,
        impexPath
    );

    var productFeedId = snapchatSettings.custom.productFeedId;
    if (!productFeedId) {
        return new Status(Status.ERROR, 'ERROR', 'Product Feed ID does not exist in custom object');
    }

    var dir = new File(baseDirPath);
    if (!dir.exists() && !dir.mkdirs()) {
        return new Status(Status.ERROR, 'ERROR', StringUtils.format('directory ({0}) does not exist', baseDirPath));
    }

    // establish processed directory
    var processedDirPath = baseDirPath + File.SEPARATOR + 'processed';
    var processedDir = new File(processedDirPath);
    if (!processedDir.exists() && !processedDir.mkdirs()) {
        return new Status(Status.ERROR, 'ERROR', StringUtils.format('could not create directory ({0})', processedDirPath));
    }
    var processedUnknownDirPath = processedDirPath + File.SEPARATOR + 'unknown';
    var processedUnknownDir = new File(processedUnknownDirPath);
    if (!processedUnknownDir.exists() && !processedUnknownDir.mkdirs()) {
        return new Status(Status.ERROR, 'ERROR', StringUtils.format('could not create directory ({0})', processedUnknownDirPath));
    }

    // process each file
    var fileList = fileHelper.getSortedFiles(baseDirPath, 'DESC', filePattern, true);
    if (fileList.size() > 0) {
        var fileListIter = fileList.iterator();
        var currentFeedUploadId = null;
        var firstRun = true;
        while (fileListIter.hasNext()) {
            var file = fileListIter.next();
            var feedUploadStatus;

            // check status of the last file upload. do not attempt to create a new file upload until the previous one is complete
            if (firstRun && !empty(snapchatSettings.custom.feedUploadId)) {
                feedUploadStatus = checkFeedUploadStatus(snapchatSettings, snapchatSettings.custom.feedUploadId, dir, processedDir);
                if (feedUploadStatus.isError()) {
                    return feedUploadStatus;
                }
            }
            firstRun = false;

            // make sure the file is still there and that it wasn't processed above
            if (!checkFileContents(file)) {
                continue; // eslint-disable-line no-continue
            }

            var fileName = file.name;
            var updateType = getUpdateType(fileName);
            if (updateType) {
                var feedUploadUrl = webDavFilePath + File.SEPARATOR + fileName;
                // make web service call to Snapchat to create the feed upload
                currentFeedUploadId = createFeedUpload(snapchatSettings, feedUploadUrl, updateType);
                if (currentFeedUploadId) {
                    // check status of the file upload. do not attempt to create a new file upload until the previous one is complete
                    feedUploadStatus = checkFeedUploadStatus(snapchatSettings, currentFeedUploadId, dir, processedDir);
                    // do not exit in error if this is the last file to process, it will get moved to the process directory on the next job run
                    if (feedUploadStatus.isError() && fileListIter.hasNext()) {
                        return feedUploadStatus;
                    }
                }
            } else {
                // move file to unknown processed folder
                fileHelper.moveFile(file, processedUnknownDir);
            }
        }
    }

    return new Status(Status.OK, 'OK');
};
