'use strict';

var CustomObjectMgr = require('dw/object/CustomObjectMgr');
var Logger = require('dw/system/Logger');
var Transaction = require('dw/system/Transaction');
var constants = require('int_snapchat/cartridge/scripts/SnapchatConstants');

/**
 * Returns the Snapchat settings custom object, if it exists.
 * If it does not exist, then it creates a new custom object instance and return it
 *
 * @returns {dw/object/CustomObject} snapchat custom object
 */
function getCustomObject() {
    var co = CustomObjectMgr.getCustomObject(constants.SOCIAL_CHANNEL_CUSTOM_OBJECT_DEFINITION, constants.SNAPCHAT_CUSTOM_OBJECT_ID);
    if (co) {
        return co;
    }

    return Transaction.wrap(function () {
        return CustomObjectMgr.createCustomObject(constants.SOCIAL_CHANNEL_CUSTOM_OBJECT_DEFINITION, constants.SNAPCHAT_CUSTOM_OBJECT_ID);
    });
}

/**
 * Clears the Snapchat values from the custom object so that we can start the process again
 * @param {dw/object/CustomObject} snapchatSettings The custom object to clear
 */
function clearValues(snapchatSettings) {
    var valuesToClear = {
        custom: {
            acceptTerms: false,
            accessToken: '',
            advertiserId: '',
            appId: '',
            appSecret: '',
            bcId: '',
            bizMngrAccessKeyWebdav: '',
            bizMngrUsername: '',
            catalogId: '',
            catalogOverview: '',
            enableAdvancedMatchingEmail: false,
            enableAdvancedMatchingPhone: false,
            externalBusinessId: '',
            externalData: '',
            externalDataKey: '',
            pixelCode: '',
            productFeedId: '',
            refreshToken: '',
            shopperClientId: '',
            shopperClientSecret: '',
            webDavClientId: '',
            webDavClientSecret: ''
        }
    };
    Transaction.wrap(function () {
        Object.keys(valuesToClear).forEach(function (key) {
            if (Object.hasOwnProperty.call(valuesToClear, key)) {
                var objectAttrDef = snapchatSettings.describe().getSystemAttributeDefinition(key);
                if (objectAttrDef) {
                    snapchatSettings[key] = valuesToClear[key];
                }
            }
        });

        if (Object.keys(valuesToClear.custom).length) {
            Object.keys(valuesToClear.custom).forEach(function (key) {
                if (Object.hasOwnProperty.call(valuesToClear.custom, key)) {
                    var objectAttrDef = snapchatSettings.describe().getCustomAttributeDefinition(key);
                    if (objectAttrDef) {
                        snapchatSettings.custom[key] = valuesToClear.custom[key];
                    }
                }
            });
        }
    });
}

/**
 * Clears the Snapchat values from the custom object so that we can start the process again
 * @param {dw.object.CustomObject} snapchatSettings The custom object to clear
 * @param {Object} valuesToSave the values to save to the custom object
 */
function saveValues(snapchatSettings, valuesToSave) {
    if (!snapchatSettings || !valuesToSave || !Object.keys(valuesToSave).length) return;

    Transaction.wrap(function () {
        Object.keys(valuesToSave).forEach(function (key) {
            if (Object.hasOwnProperty.call(valuesToSave, key)) {
                var objectAttrDef = snapchatSettings.describe().getSystemAttributeDefinition(key);
                if (objectAttrDef && valuesToSave[key] != null) {
                    snapchatSettings[key] = valuesToSave[key];
                }
            }
        });

        if (Object.keys(valuesToSave.custom).length) {
            Object.keys(valuesToSave.custom).forEach(function (key) {
                if (Object.hasOwnProperty.call(valuesToSave.custom, key)) {
                    var objectAttrDef = snapchatSettings.describe().getCustomAttributeDefinition(key);
                    if (objectAttrDef && valuesToSave.custom[key] != null) {
                        snapchatSettings.custom[key] = valuesToSave.custom[key];
                    }
                }
            });
        }
    });
}

/**
 * Removes the given custom object
 *
 * @param {dw/object/CustomObject} snapchatSettings The custom object to remove
 */
function removeCustomObject(snapchatSettings) {
    Transaction.wrap(function () {
        CustomObjectMgr.remove(snapchatSettings);
    });
}

/**
 * Parse external data from snapchatSettings.custom.externalData
 * @param {Object} snapchatSettings - custom object
 * @return {Object} externalData - parsed external data
 */
function getExternalData(snapchatSettings) {
    if (!snapchatSettings || !snapchatSettings.custom.externalData) return null;

    var externalData = null;
    try {
        externalData = JSON.parse(snapchatSettings.custom.externalData);
    } catch (e) {
        Logger.error(e.toString() + ' in ' + e.fileName + ':' + e.lineNumber);
    }
    return externalData;
}

/**
 * Fills the form with the values from the custom object
 * @param {Object} snapchatSettings - custom object
 */
function fillFormFromCustomObject(snapchatSettings) {
    if (!snapchatSettings) {
        snapchatSettings = { custom: {} }; // eslint-disable-line no-param-reassign
    }

    var form = session.forms.snapchat;
    form.clearFormElement();

    var customValuesToSet = [
        'appId',
        'appSecret',
        'bizMngrUsername',
        'bizMngrAccessKeyWebdav',
        'webDavClientId',
        'webDavClientSecret',
        'shopperClientId',
        'shopperClientSecret',
        'externalBusinessId'
    ];
    customValuesToSet.forEach(function (key) {
        if (Object.hasOwnProperty.call(form, key)) {
            form[key].value = snapchatSettings.custom[key] || '';
        }
    });

    var externalData = getExternalData(snapchatSettings);
    var externalValuesToSet = [
        'orgId',
        'shortCode',
        'catalogName',
        'catalogCurrencyCode',
        'productFeedName'
    ];
    if (externalData && Object.keys(externalData).length) {
        externalValuesToSet.forEach(function (key) {
            if (Object.hasOwnProperty.call(externalData, key)) {
                form[key].value = externalData[key] || '';
            }
        });
    } else {
        externalValuesToSet.forEach(function (key) {
            form[key].value = '';
        });
    }
}

module.exports = {
    getExternalData: getExternalData,
    getCustomObject: getCustomObject,
    removeCustomObject: removeCustomObject,
    clearValues: clearValues,
    saveValues: saveValues,
    fillFormFromCustomObject: fillFormFromCustomObject
};
