/**
 * Service Helper for Instagram integration services.
 */

'use strict';

var ArrayList = require('dw/util/ArrayList');
var HTTPRequestPart = require('dw/net/HTTPRequestPart');
var LocalServiceRegistry = require('dw/svc/LocalServiceRegistry');
var serviceHelpers = require('*/cartridge/scripts/social/helpers/serviceHelpers');

/**
 * Use this method to get countryCode from Current Locale
 *
 * @return {string} The country code found in the request, if any exists
 */
function getCountryCodeFromCurrentLocale() {
    if (empty(request.locale)) {
        return '';
    }

    var currentLocaleParsed = request.locale.split('_');
    return currentLocaleParsed[currentLocaleParsed.length > 1 ? 1 : 0];
}

/**
 * Get existing (configured in BM) service ID according to current Site and Country
 *
 * @param  {string} name The ID of the service to use while fetching the services
 * @param  {dw.svc.serviceCallback} serviceCallback The serviceCallback to use to create the service
 *
 * @return {string} The service ID configured, or undefined if no service has been found
 */
function getServiceID(name, serviceCallback) {
    var Logger = require('dw/system/Logger').getLogger('InstagramServiceHelper', 'getServiceID');
    var siteID = require('dw/system/Site').getCurrent().getID().toLowerCase();
    var countryID = getCountryCodeFromCurrentLocale().toLowerCase();
    serviceCallback = serviceCallback || {}; // eslint-disable-line no-param-reassign
    var possibleIDs = [
        name + '.' + siteID + '.' + countryID,
        name + '.' + siteID,
        name + '.' + countryID,
        name
    ];

    var existingIDs = possibleIDs.filter(function (id) {
        try {
            return !empty(LocalServiceRegistry.createService(id, serviceCallback));
        } catch (e) {
            Logger.debug('Requested Service is not configured: {0}. {1}', id, e);
            return false;
        }
    });

    return !empty(existingIDs) ? existingIDs[0] : undefined;
}

var serviceCallback = {
    createRequest: function (service, params) {
        service.setRequestMethod(params.method);
        if (params.path) {
            service.setURL(service.getURL() + params.path);
        }
        if (params.headers) {
            Object.keys(params.headers).forEach(function (key) {
                service.addHeader(key, params.headers[key]);
            });
        }
        if (params.params) {
            Object.keys(params.params).forEach(function (key) {
                service.addParam(key, params.params[key]);
            });
        }
        if (params.sendAsMultiPart && Array.isArray(params.body) && !(params.body instanceof ArrayList)) {
            var multiPartArray = new ArrayList();
            for (var idx = 0; idx < params.body.length; idx++) {
                multiPartArray.add(params.body[idx]);
            }
            params.body = multiPartArray;
        }

        // Need to check for object because stringify will escape quotes and invalidate login request
        var payload = params.body === null || params.body === '' || typeof (params.body) === 'string' || params.sendAsMultiPart ? params.body : JSON.stringify(params.body);
        return payload;
    },
    parseResponse: function (service, response) {
        return response;
    },
    filterLogMessage: function (data) {
        try {
            var logObj = JSON.parse(data);
            var result = serviceHelpers.iterate(logObj);
            return result ? JSON.stringify(result) : data;
        } catch (ex) {
            return serviceHelpers.prepareFormLogData(data);
        }
    }
};

/**
 * Returns a LocalServiceRegistry for passed service name.
 * @param {string} serviceName service name
 * @returns {dw.svc.LocalServiceRegistry} Local Service Registry
 */
function getService(serviceName) {
    return LocalServiceRegistry.createService(getServiceID(serviceName, serviceCallback), serviceCallback);
}

/**
 * Returns a HTTPRequestPart file for multiPart calls.
 * @param {string} partName part name
 * @param {dw.io.File} partFile part file
 * @returns {dw.net.HTTPRequestPart} HTTPRequestPart
 */
function convertToMultiPartFile(partName, partFile) {
    var partExtension = partFile.getName()[0] === '.' ? '' : partFile.getName().split('.').pop();
    var partEncoding = 'UTF-8';
    var partType;

    // Supporting only useful formats for integrating feed files
    switch (partExtension.toLowerCase()) {
        case 'csv': partType = 'text/csv'; break;
        case 'gzip': partType = 'application/gzip'; partEncoding = null; break;
        case 'tsv': partType = 'text/csv'; break;
        case 'xml': partType = 'text/xml'; break;
        case 'zip': partType = 'application/zip'; partEncoding = null; break;
        default: partType = 'text/plain'; break;
    }

    return new HTTPRequestPart(partName, partFile, partType, partEncoding, partFile.getName());
}

/**
 * Returns a HTTPRequestPart parameter for multiPart calls.
 * @param {string} partName part name
 * @param {string} partValue part value
 * @returns {dw.net.HTTPRequestPart} HTTPRequestPart
 */
function convertToMultiPartParam(partName, partValue) {
    return new HTTPRequestPart(partName, partValue);
}

module.exports = {
    getService: getService,
    convertToMultiPartFile: convertToMultiPartFile,
    convertToMultiPartParam: convertToMultiPartParam
};
