'use strict';

var HTTPClient = require('dw/net/HTTPClient');
var Logger = require('dw/system/Logger').getLogger('Instagram', 'instagramSfccService');
var Site = require('dw/system/Site');

var instagramContants = require('int_instagram/cartridge/scripts/InstagramConstants');
var validationHelper = require('bm_socialchannels/cartridge/scripts/utils/validationHelper');

/**
 * Get a new access token from Account Manager
 *
 * @param {string} clientId Client ID
 * @param {string} clientSecret Client Secret
 * @param {string} orgId Organization ID
 * @param {string} scopes the scope list to ask for permissions
 * @returns {Object} the access token
 */
function getAMAccessToken(clientId, clientSecret, orgId, scopes) {
    try {
        var tenantId = orgId && orgId.length > 7 ? orgId.substring(7) : null;

        if (empty(clientId) || empty(clientSecret) || empty(tenantId) || empty(scopes)) {
            Logger.debug('getAMAccessToken: missing or malformed param - clientId {0}, secret, orgId {1} or scopes {2}', clientId, orgId, scopes);
            return { error: true, errorDescription: 'Missing or malformed params', token: null };
        }

        var httpClient = new HTTPClient();
        httpClient.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
        httpClient.open('POST', 'https://account.demandware.com/dwsso/oauth2/access_token', clientId, clientSecret);
        httpClient.send([
            'grant_type=client_credentials',
            'scope=SALESFORCE_COMMERCE_API:' + tenantId + ' ' + scopes
        ].join('&'));

        if (httpClient.statusCode === 200) {
            return { error: false, token: JSON.parse(httpClient.text).access_token };
        }

        var errorObject = !empty(httpClient.errorText) ? JSON.parse(httpClient.errorText) : null;
        var errorDescription = 'Token could not be obtained';

        if (errorObject && (errorObject.error === 'invalid_client' || errorObject.error === 'invalid_scope')) {
            errorDescription = errorObject.error_description;
        }

        return { error: true, errorDescription: errorDescription, token: null };
    } catch (error) {
        Logger.error(error.toString() + ' in ' + error.fileName + ':' + error.lineNumber);
        return { error: true, errorDescription: error.message, token: null };
    }
}

/**
 * Get all catalog ids from instance
 *
 * @param {dw/object/CustomObject} instagramSettings the Instagram settings custom object instance
 * @returns {Object} an object containing the response or error if any happened
 */
function getCatalogs(instagramSettings) {
    try {
        if (empty(instagramSettings)) {
            return null;
        }

        var clientId = instagramSettings.custom.amClientId;
        var clientSecret = instagramSettings.custom.amClientSecret;
        var shortCode = instagramSettings.custom.shortCode;
        var orgId = instagramSettings.custom.orgId;

        if (empty(clientId) || empty(clientSecret) || empty(shortCode) || empty(orgId)) {
            Logger.warn('getCatalogs: missing param - clientId {0}, secret, shortCode {1} or orgId {2}', clientId, shortCode, orgId);
            return null;
        }

        var responseToken = getAMAccessToken(clientId, clientSecret, orgId, 'sfcc.catalogs.rw');

        if (empty(responseToken) || responseToken.error || empty(responseToken.token)) {
            Logger.warn('getCatalogs: token could not be obtained - response {0}', !empty(responseToken) ? JSON.stringify(responseToken) : null);
            return null;
        }

        var canRetry = true;
        var catalogIds = [];
        var catalogTotal = 100; // starts with quota value then update to the real value
        var limit = 50; // maximum records to retrieve per request, not to exceed 50
        var offset = 0; // used to retrieve the results based on a particular resource offset
        var httpClient = new HTTPClient();

        while (offset < catalogTotal) {
            httpClient.setRequestHeader('Authorization', 'Bearer ' + responseToken.token);
            httpClient.open('GET', 'https://' + shortCode + '.api.commercecloud.salesforce.com/product/catalogs/v1/organizations/' + orgId + '/catalogs?limit=' + limit + '&offset=' + offset);
            httpClient.send();

            if (httpClient.statusCode === 200 && !empty(httpClient.text)) {
                var catalogsResponse = JSON.parse(httpClient.text);
                catalogTotal = catalogsResponse.total;
                offset += limit;
                catalogIds = catalogIds.concat(catalogsResponse.data.map(function (c) { return c.id; }));
                canRetry = true;
            } else {
                if (!canRetry) {
                    Logger.error('getCatalogs: catalog list could not be obtained - status {0}', httpClient.statusCode);
                    return null;
                }

                canRetry = false;
            }
        }

        return catalogIds;
    } catch (error) {
        Logger.error(error.toString() + ' in ' + error.fileName + ':' + error.lineNumber);
        return null;
    }
}

/**
 * Validates Account Manager credentials by trying to get an access token
 *
 * @param {string} clientId AM Client ID to test
 * @param {string} clientSecret AM Client Secret to test
 * @param {string} orgId Organization ID to test
 * @returns {Object} validation object
 */
function validateAMCredentials(clientId, clientSecret, orgId) {
    if (empty(clientId) || clientId === 'aaaaaaaaaaaaaaaaaaaaaaaaaaaaaa' || empty(clientSecret) || empty(orgId)) {
        return false;
    }

    var validation = getAMAccessToken(clientId, clientSecret, orgId, instagramContants.INSTAGRAM_DEFAULT_SCOPES);

    return validation;
}

/**
 * Validates SLAS credentials by trying to get an access token
 *
 * @param {string} shopperClientId SLAS Client ID to test
 * @param {string} shopperClientSecret SLAS Client Secret to test
 * @param {string} shortCode Short Code to test
 * @param {string} orgId Organization ID to test
 * @returns {Object} validation pass = true|false
 */
function validatePrivateTSOBClient(shopperClientId, shopperClientSecret, shortCode, orgId) {
    if (empty(shopperClientId) || empty(shopperClientSecret) || empty(shortCode) || empty(orgId)) {
        return false;
    }

    var validation = validationHelper.validatePrivateTSOBClient(shopperClientId, shopperClientSecret, Site.getCurrent().getID(), shortCode, orgId);

    return !empty(validation) && validation.error === false;
}

module.exports = {
    getCatalogs: getCatalogs,
    validateAMCredentials: validateAMCredentials,
    validatePrivateTSOBClient: validatePrivateTSOBClient
};
