'use strict';

var Logger = require('dw/system/Logger').getLogger('Instagram', 'instagramService');
var Site = require('dw/system/Site');
var StringUtils = require('dw/util/StringUtils');
var System = require('dw/system/System');

var serviceHelper = require('./serviceHelper');
var constants = require('../InstagramConstants');

var CONTENT_TYPE = 'application/json';
var counter = 0;

/**
 * Get the current system user access token to be used
 *
 * @param {dw/object/CustomObject} instagramSettings the Instagram settings custom object instance
 * @returns {Object} the access token
 */
function getAccessToken(instagramSettings) {
    if (empty(instagramSettings)) {
        return null;
    }

    return instagramSettings.custom.userToken || '';
}

/**
 * Parses the response
 *
 * @param {Object} result the response result
 * @param {string} errorCode an error code for the response
 * @returns {Object} an object containing the response or error if any happened
 */
function parseResponse(result, errorCode) {
    try {
        if (!result.ok) {
            throw new Error(result.errorMessage);
        }

        var resultText = JSON.parse(result.object.text);
        Logger.info(counter++ + ': result: ' + result.object.text + '\n');

        return {
            error: false,
            result: resultText
        };
    } catch (err) {
        Logger.error('Error occurred while {0}. Error Message: {1}', errorCode.replace('.', ' ', 'g'), err.message);
    }

    return {
        error: true,
        errorCode: errorCode
    };
}

/**
 * Uninstall Meta Business Extension from environment
 * https://developers.facebook.com/docs/facebook-business-extension/fbe/guides/uninstall#uninstall-fbe
 *
 * @param {dw/object/CustomObject} instagramSettings the Instagram settings custom object instance
 * @returns {Object} an object containing the response or error if any happened
 */
function disconnectMBE(instagramSettings) {
    var service = serviceHelper.getService(constants.SERVICES.INSTAGRAM.BASE);
    var params = {
        method: 'DELETE',
        path: StringUtils.format(constants.ENDPOINTS.MBE_CONNECTION, instagramSettings.custom.externalBusinessId, getAccessToken(instagramSettings))
    };

    var result = service.call(params);
    return parseResponse(result, 'disconnect.mbe.call');
}

/**
 * Get Meta Integration Commerce Extension (MICE) Url
 * @param {dw/object/CustomObject} instagramSettings the Instagram settings custom object instance
 * @returns {Object} an object containing the response or error if any happened
 */
function getCommerceExtensionUrl(instagramSettings) {
    var service = serviceHelper.getService(constants.SERVICES.INSTAGRAM.BASE);
    var params = {
        method: 'GET',
        path: StringUtils.format(constants.ENDPOINTS.COMMERCE_EXTENSION, instagramSettings.custom.externalBusinessId, getAccessToken(instagramSettings))
    };

    var result = service.call(params);
    return parseResponse(result, 'commerce.extension.call');
}

/**
 * Send feed notification
 * @param {dw/object/CustomObject} instagramSettings the Instagram settings custom object instance
 * @param {string} feedURL the feed URL
 * @param {string} feedType the feed type
 * @param {string} updateType the update type
 * @param {number} updateTime the update time
 * @returns {boolean} true if the upload process succeed, false otherwise
 */
function notifyFeed(instagramSettings, feedURL, feedType, updateType, updateTime) {
    var service = serviceHelper.getService(constants.SERVICES.INSTAGRAM.BASE);
    var params = {
        method: 'POST',
        path: StringUtils.format(constants.ENDPOINTS.FEED_NOTIFICATION, instagramSettings.custom.commercePartnerIntegrationId),
        headers: {
            'Content-Type': CONTENT_TYPE
        },
        body: {
            access_token: getAccessToken(instagramSettings),
            url: feedURL,
            feed_type: feedType,
            update_type: updateType,
            update_time: updateTime
        }
    };

    // retry variables
    var retryLimit = constants.SERVICE_RETRY_COUNT;
    var retryCount = 0;
    var success = false;
    var result;

    while (!success && retryCount < retryLimit) {
        try {
            Logger.info('retryCount: ' + (retryCount + 1) + ' | Request:' + JSON.stringify(params));
            result = service.call(params);
            if (result.ok !== true) {
                if (result.errorMessage != null) {
                    Logger.info('retryCount: ' + (retryCount + 1) + ' | Error occurred calling notify service: ' + result.errorMessage);
                }
            } else if (result.object && result.object.text) {
                var response = JSON.parse(result.object.text);
                if (response && response.success !== true) {
                    Logger.info('retryCount: ' + (retryCount + 1) + ' | Error in notify service response: ' + result.object.text);
                } else {
                    Logger.info('retryCount: ' + (retryCount + 1) + ' | Notify service response: ' + result.object.text);
                    success = true;
                }
            }
            retryCount++;
        } catch (ex) {
            retryCount++;
            if (retryCount >= retryLimit) {
                // If we've reached the maximum retry limit, return not success
                return success;
            }
        }
    }
    return success;
}

/**
 * Send post onboarding data to Meta
 *
 * @param {dw/object/CustomObject} instagramSettings the Instagram settings custom object instance
 * @returns {Object} an object containing the response or error if any happened
 */
function postOnboardingData(instagramSettings) {
    var service = serviceHelper.getService(constants.SERVICES.INSTAGRAM.BASE);
    var params = {
        method: 'POST',
        path: StringUtils.format(constants.ENDPOINTS.POST_ONBOARDING, instagramSettings.custom.commercePartnerIntegrationId),
        headers: {
            'Content-Type': CONTENT_TYPE
        },
        body: {
            access_token: getAccessToken(instagramSettings),
            org_id: instagramSettings.custom.orgId,
            short_code: instagramSettings.custom.shortCode,
            site_id: Site.getCurrent().getID(),
            product_full_feed_job_id: instagramSettings.custom.feedUploadId,
            shop_domain: System.getInstanceHostname(),
            instance_hostname: instagramSettings.custom.hostname,
            client_id: instagramSettings.custom.amClientId,
            client_secret: instagramSettings.custom.amClientSecret,
            slas_id: instagramSettings.custom.shopperClientId,
            slas_secret: instagramSettings.custom.shopperClientSecret
        }
    };

    var result = service.call(params);
    return parseResponse(result, 'post.onboarding.call');
}

/**
 * Validate Meta Business Extension System User Access Token
 *
 * @param {dw/object/CustomObject} instagramSettings the Instagram settings custom object instance
 * @returns {Object} an object containing the response or error if any happened
 */
function validateAccessToken(instagramSettings) {
    var service = serviceHelper.getService(constants.SERVICES.INSTAGRAM.BASE);
    var params = {
        method: 'GET',
        path: StringUtils.format(constants.ENDPOINTS.VALIDATE_TOKEN, getAccessToken(instagramSettings))
    };

    var result = service.call(params);
    return parseResponse(result, 'validate.token.call');
}

module.exports = {
    disconnectMBE: disconnectMBE,
    getCommerceExtensionUrl: getCommerceExtensionUrl,
    notifyFeed: notifyFeed,
    postOnboardingData: postOnboardingData,
    validateAccessToken: validateAccessToken
};
