'use strict';

var File = require('dw/io/File');
var Status = require('dw/system/Status');
var StringUtils = require('dw/util/StringUtils');
var System = require('dw/system/System');

var constants = require('int_instagram/cartridge/scripts/InstagramConstants');
var customObjectHelper = require('int_instagram/cartridge/scripts/customObjectHelper');
var fileHelpers = require('*/cartridge/scripts/social/helpers/fileHelpers');
var instagramService = require('int_instagram/cartridge/scripts/services/instagramService');

/**
 * Notifies Instagram about the successful feed generation via API call
 * @param {Array} args job arguments
 * @param {dw.job.JobStepExecution} stepExecution step Execution
 * @returns {dw.system.Status} Exit status for a job run
 */
exports.execute = function (args) {
    var StepHelper = require('int_instagram/cartridge/scripts/stepHelper');
    var instagramSettings = customObjectHelper.getSettings();

    // Get Feed Type
    var feedType = args.FeedType ? args.FeedType.toUpperCase() : '';
    // Get Update Type
    var updateType = args.UpdateType ? args.UpdateType.toUpperCase() : '';

    // Get if we should compress files before notify
    var compressBeforeNotify = args.CompressBeforeNotify;

    var instanceHostname = System.getInstanceHostname();
    var notifyFile;
    var svcResult;

    // List of notification responses
    var notifyResponses = [];

    // If notification has errors set true
    var notificationHasErrors = false;

    // Standard responses for each file to be notified
    var RESPONSES = {
        SUCCESS: 'Notified Successfully',
        FAILED: 'Notification Service Failed'
    };

    //  Is the current jobStep being skipped?  If so, exit early
    if (StepHelper.isDisabled(args)) {
        return new Status(Status.OK, 'SKIP', 'Step disabled, skip it...');
    }

    //  MBE is not connected
    if (empty(instagramSettings.custom.userToken)) {
        return new Status(Status.OK, 'SKIP', 'Onboarding not completed...');
    }

    if (empty(feedType)) {
        return new Status(Status.ERROR, 'Error', 'Feed type parameter not found');
    }

    if (empty(updateType)) {
        return new Status(Status.ERROR, 'Error', 'Update type parameter not found');
    }

    var path = File.IMPEX;

    if (!empty(args.FileFolder)) {
        if (args.FileFolder[0] !== File.SEPARATOR) {
            args.FileFolder = File.SEPARATOR + args.FileFolder;
        }
        path += args.FileFolder;
    }

    // Get Instagram notify file
    var filesToExport = fileHelpers.getFilesToExport(path);

    if (compressBeforeNotify) {
        var compressedFilesToExport = [];
        for (let i = 0; i < filesToExport.length; i++) {
            var file = filesToExport[i];
            if (fileHelpers.endsWith(file.fullPath, '.zip')) {
                compressedFilesToExport.push(file);
            } else {
                try {
                    var compressedFile = new File(file.fullPath.concat('.zip'));
                    file.zip(compressedFile);
                    compressedFilesToExport.push(compressedFile);
                    file.remove();
                } catch (error) {
                    compressedFilesToExport.push(file);
                }
            }
        }
        filesToExport = compressedFilesToExport;
    }

    // Delete .tracking file if it exists (removing it asap will avoid duplicating notifications)
    fileHelpers.deleteTrackingFile(path);

    for (var idx = 0; idx < filesToExport.length; idx++) {
        notifyFile = filesToExport[idx];

        var fullPath = StringUtils.format(
            'https://{0}{1}{2}',
            instanceHostname,
            constants.WEBDAV_BASE_PATH,
            notifyFile.getFullPath()
        );

        svcResult = instagramService.notifyFeed(instagramSettings, fullPath, feedType, updateType, notifyFile.lastModified());

        if (!svcResult) {
            notifyResponses.push({ fileName: notifyFile.getFullPath(), status: RESPONSES.FAILED });
            notificationHasErrors = true;
        } else {
            notifyResponses.push({ fileName: notifyFile.getFullPath(), status: RESPONSES.SUCCESS });
        }
    }

    if (notificationHasErrors) {
        return new Status(Status.ERROR, 'ERROR', JSON.stringify(notifyResponses));
    }

    return new Status(Status.OK, 'OK', JSON.stringify(notifyResponses));
};
