'use strict';

/**
 * @module scripts/jobs/categories
 */

var Logger = require('dw/system/Logger').getLogger('ExportListOfCategories');
var ObjectAttributeDefinition = require('dw/object/ObjectAttributeDefinition');
var StringUtils = require('dw/util/StringUtils');

var customObjectHelper = require('int_instagram/cartridge/scripts/customObjectHelper');
var instagramSfccService = require('int_instagram/cartridge/scripts/services/instagramSfccService');

var blackList = ['displayMode', 'searchPlacement', 'searchRank', 'siteMapChangeFrequency', 'siteMapIncluded', 'siteMapPriority', 'template'];
var customAttributes;
var systemAttributes;

/**
 * Format the value of a category field
 *
 * @param {string} field field name
 * @param {string} value field value
 * @param {Object} valueType field value type
 * @returns {string} field formatted
 */
function formatField(field, value, valueType) {
    var isEnum = valueType === ObjectAttributeDefinition.VALUE_TYPE_ENUM_OF_INT
        || valueType === ObjectAttributeDefinition.VALUE_TYPE_ENUM_OF_STRING;

    var isSet = valueType === ObjectAttributeDefinition.VALUE_TYPE_SET_OF_INT
        || valueType === ObjectAttributeDefinition.VALUE_TYPE_SET_OF_NUMBER
        || valueType === ObjectAttributeDefinition.VALUE_TYPE_SET_OF_STRING;

    if (isSet !== false) {
        return '';
    }
    if (empty(value) || (isEnum && empty(value.displayValue))) {
        return ', ';
    }
    if (valueType === ObjectAttributeDefinition.VALUE_TYPE_IMAGE) {
        return value.httpsURL + ', ';
    }

    return typeof value === 'string' ? value.replace(/,?\s+/g, ' ') + ', ' : value + ', ';
}

/**
 * Get additional category fields
 * @param {string} parentID category parent ID
 * @param {string} categoryID category ID
 * @param {string} catalogID catalog ID
 * @param {boolean} isPrimaryCatalog is part of the site primary catalog
 * @returns {Object} object with additional information
 */
function getAdditionalFields(parentID, categoryID, catalogID, isPrimaryCatalog) {
    var URLUtils = require('dw/web/URLUtils');

    if (empty(parentID)) {
        parentID = ''; // eslint-disable-line
    }
    if (empty(categoryID)) {
        categoryID = ''; // eslint-disable-line
    }
    if (empty(catalogID)) {
        catalogID = ''; // eslint-disable-line
    }

    return {
        header: 'Parent ID, URL, Catalog ID, Is Primary Catalog',
        line: parentID + ', ' + URLUtils.https('Search-Show', 'cgid', categoryID || '') + ', ' + catalogID + ', ' + isPrimaryCatalog
    };
}

/**
 * Write a header with the category attributes, ignoring SET_* types
 *
 * @param {dw/io/FileWriter} writer fileWriter
 */
function writeHeader(writer) {
    var textHeader = '';
    var valueType;
    var isSet;

    for (var i = 0; i < systemAttributes.length; i++) {
        valueType = systemAttributes[i].valueTypeCode;
        isSet = valueType === ObjectAttributeDefinition.VALUE_TYPE_SET_OF_INT
        || valueType === ObjectAttributeDefinition.VALUE_TYPE_SET_OF_NUMBER
        || valueType === ObjectAttributeDefinition.VALUE_TYPE_SET_OF_STRING;

        if (!isSet) {
            textHeader += systemAttributes[i].displayName + ', ';
        }
    }
    for (var j = 0; j < customAttributes.length; j++) {
        valueType = customAttributes[j].valueTypeCode;
        isSet = valueType === ObjectAttributeDefinition.VALUE_TYPE_SET_OF_INT
        || valueType === ObjectAttributeDefinition.VALUE_TYPE_SET_OF_NUMBER
        || valueType === ObjectAttributeDefinition.VALUE_TYPE_SET_OF_STRING;

        if (!isSet) {
            textHeader += customAttributes[j].displayName + ', ';
        }
    }

    textHeader += getAdditionalFields().header;

    writer.writeLine(textHeader);
}

/**
 * Write a line with the category values
 *
 * @param {Object} category category object
 * @param {string} parentID parent category name
 * @param {Object} catalog catalog object
 * @param {string} primaryCatalogID primary catalog ID
 * @param {dw/io/FileWriter} writer fileWriter
 */
function writeCategoryLine(category, parentID, catalog, primaryCatalogID, writer) {
    var textWrite = '';

    for (var i = 0; i < systemAttributes.length; i++) {
        var fieldID = systemAttributes[i].ID;
        textWrite += formatField(fieldID, category[fieldID], systemAttributes[i].valueTypeCode);
    }
    for (var j = 0; j < customAttributes.length; j++) {
        var fieldIDcustom = customAttributes[j].ID;
        textWrite += formatField(fieldIDcustom, category.custom[fieldIDcustom], customAttributes[j].valueTypeCode);
    }

    textWrite += getAdditionalFields(parentID, category.ID, catalog.ID, catalog.ID === primaryCatalogID).line;

    writer.writeLine(textWrite);

    if (!empty(category.subCategories) && category.subCategories.length) {
        for (var idxSub = 0; idxSub < category.subCategories.length; idxSub++) {
            var subcategory = category.subCategories[idxSub];
            writeCategoryLine(subcategory, category.ID, catalog, primaryCatalogID, writer);
        }
    }
}

/**
 * Export categories to Meta
 *
 * @param {Object} parameters The job's parameters
 * @returns {dw/system/Status} The status of the job
 */
function exportCategories(parameters) {
    var CatalogMgr = require('dw/catalog/CatalogMgr');
    var File = require('dw/io/File');
    var FileWriter = require('dw/io/FileWriter');
    var Site = require('dw/system/Site');
    var Status = require('dw/system/Status');
    var StepHelper = require('int_instagram/cartridge/scripts/stepHelper');
    var fileHelpers = require('*/cartridge/scripts/social/helpers/fileHelpers');
    var error = false;

    //  Is the current jobStep being skipped?  If so, exit early
    if (StepHelper.isDisabled(parameters)) {
        return new Status(Status.OK, 'SKIP', 'Step disabled, skip it...');
    }

    try {
        var currentSite = Site.getCurrent();
        var path = File.IMPEX;
        var primaryCatalogID = CatalogMgr.getSiteCatalog().ID;
        var timeStamp = new Date().getTime();

        if (!empty(parameters.FolderPath)) {
            if (parameters.FolderPath[0] !== File.SEPARATOR) {
                parameters.FolderPath = File.SEPARATOR + parameters.FolderPath;
            }
            path += parameters.FolderPath;
        }

        var dir = new File(path);

        if (!dir.exists() && !dir.mkdirs()) {
            throw new Error('The export directory could not be created.');
        }

        var locales = parameters.Locale.toLowerCase() === 'all' ? currentSite.getAllowedLocales() : [parameters.Locale];
        var catalogs = [];

        if (!empty(parameters.CatalogId) && parameters.CatalogId.trim() === '*') {
            var instagramSettings = customObjectHelper.getSettings();
            var catalogIds = instagramSfccService.getCatalogs(instagramSettings);

            if (!empty(catalogIds)) {
                catalogs = catalogIds.map(function (c) { return CatalogMgr.getCatalog(c); });
            } else {
                catalogs.push(CatalogMgr.getSiteCatalog());
            }
        } else {
            catalogs.push(CatalogMgr.getCatalog(parameters.CatalogId));
        }

        if (empty(catalogs) || catalogs[0] === null) {
            throw new Error('The catalog(s) could not be read.');
        }

        for (var idx = 0; idx < locales.length; idx++) {
            var writer;

            try {
                if (request.setLocale(locales[idx])) {
                    var filename = 'categories-' + currentSite.ID + '-' + timeStamp + '-' + locales[idx] + '.csv';
                    var file = new File(dir, filename);

                    if (!file.exists() && !file.createNewFile()) {
                        throw new Error('The export file could not be created: ' + filename);
                    }

                    writer = new FileWriter(file);

                    for (var idxCatalog = 0; idxCatalog < catalogs.length; idxCatalog++) {
                        var catalog = catalogs[idxCatalog];
                        if (!empty(catalog) && catalog.root) {
                            if (idxCatalog === 0) {
                                var describe = catalog.root.describe().getAttributeDefinitions().toArray();
                                systemAttributes = describe.filter(function (data) { return data.system === true && blackList.indexOf(data.ID) < 0; });
                                customAttributes = describe.filter(function (data) { return data.system === false; });
                                writeHeader(writer);
                            }
                            writeCategoryLine(catalog.root, '', catalog, primaryCatalogID, writer);
                        } else {
                            throw new Error(StringUtils.format('The catalog {0} could not be found for locale {1}', parameters.CatalogId, locales[idx]));
                        }
                    }

                    fileHelpers.addFileNameToTrackingFile(path, filename);
                } else {
                    throw new Error('The locale could not be defined: ' + locales[idx]);
                }
            } catch (err) {
                Logger.error(err.message);
                error = true;
            } finally {
                if (writer) {
                    writer.close();
                }
            }
        }
    } catch (e) {
        Logger.error(e.message);
        error = true;
    }

    return !error ? new Status(Status.OK, 'All exports succeed.') : new Status(Status.ERROR, 'There was an error executing the jobstep.');
}

module.exports.exportCategories = exportCategories;
