'use strict';

var Calendar = require('dw/util/Calendar');
var CustomObjectMgr = require('dw/object/CustomObjectMgr');
var Logger = require('dw/system/Logger');
var Site = require('dw/system/Site');
var StringUtils = require('dw/util/StringUtils');
var System = require('dw/system/System');
var Transaction = require('dw/system/Transaction');
var URLUtils = require('dw/web/URLUtils');

var constants = require('int_instagram/cartridge/scripts/InstagramConstants');

/**
 * Check if all required fields are filled
 *
 * @param {dw/object/CustomObject} instagramSettings the Instagram settings custom object instance
 * @returns {Object} validation result
 */
function checkRequiredFields(instagramSettings) {
    return !empty(instagramSettings)
    && !empty(instagramSettings.custom.amClientId)
    && !empty(instagramSettings.custom.amClientSecret)
    && !empty(instagramSettings.custom.hostname)
    && !empty(instagramSettings.custom.feedUploadId)
    && !empty(instagramSettings.custom.orgId)
    && !empty(instagramSettings.custom.shortCode)
    && !empty(instagramSettings.custom.shopperClientId)
    && !empty(instagramSettings.custom.shopperClientSecret);
}

/**
 * Clears information related to the MBE connection
 *
 * @param {dw/object/CustomObject} instagramSettings the Instagram settings custom object instance
 * @returns {Object} transaction result
 */
function clearSettings(instagramSettings) {
    return Transaction.wrap(function () {
        instagramSettings.custom.appId = null;
        instagramSettings.custom.businessManagerId = null;
        instagramSettings.custom.catalogId = null;
        instagramSettings.custom.commercePartnerIntegrationId = null;
        instagramSettings.custom.externalBusinessId = null;
        instagramSettings.custom.metaConnected = false;
        instagramSettings.custom.userToken = null;
    });
}

/**
 * Fills the form with custom object values
 *
 * @param {dw/object/CustomObject} instagramSettings the Instagram settings custom object instance
 */
function fillFormFromCustomObject(instagramSettings) {
    if (!instagramSettings) {
        instagramSettings = { custom: {} }; // eslint-disable-line no-param-reassign
    }

    var form = session.forms.instagram;

    if (!empty(form)) {
        form.clearFormElement();
        form.amclientid.value = instagramSettings.custom.amClientId || '';
        form.amclientsecret.value = instagramSettings.custom.amClientSecret || '';
        form.businessmanagerid.value = instagramSettings.custom.businessManagerId || '';
        form.commercepartnerid.value = instagramSettings.custom.commercePartnerIntegrationId || '';
        form.externalbusinessid.value = instagramSettings.custom.externalBusinessId || '';
        form.feeduploadid.value = instagramSettings.custom.feedUploadId || '';
        form.hostname.value = instagramSettings.custom.hostname || '';
        form.orgid.value = instagramSettings.custom.orgId || '';
        form.shortcode.value = instagramSettings.custom.shortCode || '';
        form.slasclientid.value = instagramSettings.custom.shopperClientId || '';
        form.slasclientsecret.value = instagramSettings.custom.shopperClientSecret || '';
    }
}

/**
 * Get parameters related to the current instance
 *
 * @param {dw/object/CustomObject} instagramSettings the Instagram settings custom object instance
 * @returns {Object} parameters object
 */
function getInstanceParams(instagramSettings) {
    var currency = Site.getCurrent().getDefaultCurrency();
    var hostname = System.getInstanceHostname();
    var siteID = Site.getCurrent().getID();
    var timezone = Site.getCurrent().getTimezone();
    var url = URLUtils.https('Home-Show').abs().toString();
    var timeStamp = StringUtils.formatCalendar(new Calendar(), "yyyy-MM-dd'T'HH-mm-ss-sss'Z'");
    // set unique identifier for current environment
    var externalBusinessId = hostname + '_' + siteID + '_' + timeStamp;

    var splashUrl = checkRequiredFields(instagramSettings)
        ? StringUtils.format(constants.ENDPOINTS.SPLASH_EXTENSION,
            constants.APP_ID,
            externalBusinessId,
            constants.MBE.BUSINESS_NAME,
            currency,
            encodeURIComponent(timezone),
            encodeURIComponent(JSON.stringify({
                feed_upload_id: constants.INSTAGRAM_DEFAULT_PRODUCT_FEED_JOB,
                org_id: instagramSettings.custom.orgId,
                site_id: siteID,
                short_code: instagramSettings.custom.shortCode,
                url: url
            })),
            constants.MBE.BUSINESS_VERTICAL,
            constants.MBE.CHANNEL)
        : null;

    return {
        appId: constants.APP_ID,
        externalBusinessId: externalBusinessId,
        splashUrl: splashUrl
    };
}

/**
 * Get the current onboarding step from Meta integration
 *
 * @param {dw/object/CustomObject} instagramSettings the Instagram settings custom object instance
 * @returns {Object} onboarding step
 */
function getOnboardingStep(instagramSettings) {
    if (empty(instagramSettings)) {
        return 0;
    }

    if (instagramSettings.custom.metaConnected && !empty(instagramSettings.custom.userToken)) {
        return 3;
    }

    if (!empty(instagramSettings.custom.userToken)) {
        return 2;
    }

    if (checkRequiredFields(instagramSettings)) {
        return 1;
    }

    return 0;
}

/**
 * Get the Instagram settings custom object instance
 *
 * @param {boolean} isStorefrontRequest is storefront request?
 * @returns {dw/object/CustomObject} custom object
 */
function getSettings(isStorefrontRequest) {
    var co = CustomObjectMgr.getCustomObject(constants.SOCIAL_CHANNEL_CUSTOM_OBJECT_DEFINITION, constants.INSTAGRAM_CUSTOM_OBJECT_ID);
    if (co) {
        return co;
    }

    // do not create the custom object if this is a storefront request
    if (isStorefrontRequest) {
        Logger.warn('Missing Custom Object definition');
        return null;
    }

    return Transaction.wrap(function () {
        return CustomObjectMgr.createCustomObject(constants.SOCIAL_CHANNEL_CUSTOM_OBJECT_DEFINITION, constants.INSTAGRAM_CUSTOM_OBJECT_ID);
    });
}

module.exports = {
    clearSettings: clearSettings,
    fillFormFromCustomObject: fillFormFromCustomObject,
    getInstanceParams: getInstanceParams,
    getOnboardingStep: getOnboardingStep,
    getSettings: getSettings
};
