'use strict';

const path = require('path');
const fs = require('fs');
const { DOMParser, XMLSerializer } = require('xmldom');

const {
    getSitePreferenceRequestBody,
    getSocialChannelDetails,
    restoreSiteDirectories,
    updateCartridgePath,
    updateOcapiWebDavPermissions,
    updateScapiPermissions,
    updateSitePreferences,
    upsertScapiClient,
    writeAndImportAccessRole
} = require('../utils/featureHelpers');

const {
    BIZ_MNGR_SITE_ID,
    DEPLOY_OPTIONS,
    DIRECTORIES,
    SITE_PREF_GROUPS
} = require('../utils/socialConstants');
const SCRIPT_NAME = '[SOCIAL SETUP]';

const modifyXMLAndDownload = async ({ logger }, xmlFileName, siteConfigSocialChannel, siteInfo) => {
    logger.info('Update bm_extensions.xml file to enable relevant menu actions.');
    let socialChannelStatus = new Map([
        ['instagram', false],
        ['snapchat', false],
        ['google', false],
        ['tiktok', false]]);
    // Read the XML file
    fs.readFile(xmlFileName, 'utf8', (err, data) => {
        if (err) {
            logger.debug('Error reading the file:', err);
            return;
        }

        // Parse the XML data
        let parser = new DOMParser();
        let xmlDoc = parser.parseFromString(data, 'text/xml');

        // get already installed channel list
        getInstalledSocialChannelsList({ logger }, siteInfo, socialChannelStatus);

        // Find elements with the 'site' attribute and modify them
        let elements = Array.from(xmlDoc.getElementsByTagName('menuaction')); // Get all elements

        for (let element of elements) {
            if (element.hasAttribute('site')) {
                let idValue = element.getAttribute('id');
                if (!idValue) continue; // Skip if 'id' is missing

                let idSplit = idValue.split('_');
                let siteStatus = idSplit.length > 1
                    && (idValue.toLowerCase().includes(siteConfigSocialChannel.toLowerCase())
                    || (socialChannelStatus.has(idSplit[2]) && socialChannelStatus.get(idSplit[2])));

                // Set 'site' attribute based on the condition
                element.setAttribute('site', !!siteStatus);
            }
        }

        // Serialize the XML back to a string
        let serializer = new XMLSerializer();
        let updatedXmlString = serializer.serializeToString(xmlDoc);

        // Write the modified XML back to the file
        fs.writeFile(xmlFileName, updatedXmlString, 'utf8', (err) => {
            if (err) {
                logger.debug('Error writing the file bm_extensions.xml:', err);
            } else {
                logger.info('File bm_extensions.xml successfully updated!');
            }
        });
    });
};

const getInstalledSocialChannelsList = ({ logger }, siteInfo, socialChannelStatus) => {
    console.log('Updated socialChannelStatus:', socialChannelStatus);
    logger.info('Retrieve already installed social channels in the site');
    if (siteInfo?.data?.cartridges) {
        let currentCartridges = siteInfo.data.cartridges.split(':');
        currentCartridges.forEach(cartridge => {
            if (cartridge.includes('int')) {
                let cartridgesName = cartridge.split('_');
                if (cartridgesName[1] && socialChannelStatus.has(cartridgesName[1])) {
                    socialChannelStatus.set(cartridgesName[1], true);
                }
            }
        });
    }
};

/**
 * Main entry point for social integration feature setup
 * @param {Object} args arguments
 * @param {Environment} args.env environment
 * @param {Logger} args.logger logger
 * @param {MigrationHelpers} args.helpers helpers
 * @param {Object} args.vars variables
 * @returns {Promise<void>} void
 */
module.exports = {
    setupSocialChannel: async function ({ env, logger, helpers, vars }) {
        if (!vars.socialChannel || !vars.deployOption) {
            logger.error(`${SCRIPT_NAME} Could not continue! missing at least 1 of the below environment variables.\n
                SFCC_VARS__socialChannel\n
                SFCC_VARS__deployOption`);
            return;
        }

        const { siteArchiveImport, syncCartridges } = helpers;

        // set social channel base
        vars.socialChannelBase = vars.socialChannel.split('_')[0];

        // gather cartridge and data details per social channel
        const socialChannelDetails = getSocialChannelDetails({ vars, logger });
        logger.debug(`${SCRIPT_NAME} ${JSON.stringify(socialChannelDetails)}`);

        // deploy channel specific data to the sandbox
        if (DEPLOY_OPTIONS.DATA_OPTIONS.includes(vars.deployOption)) {
            logger.info(`${SCRIPT_NAME} data deploy is enabled and starting.`);

            // ***********
            // import data folders to sandbox
            // ***********
            if (socialChannelDetails.dataFolders.length) {
                let dataFolders = socialChannelDetails.dataFolders;
                for (let i = 0; i < dataFolders.length; i++) {
                    let dataFolder = dataFolders[i];
                    let dataFolderPath = path.join(process.cwd(), DIRECTORIES.BASE_DATA_DIR, dataFolder);
                    logger.info(`${SCRIPT_NAME} importing data: "${dataFolderPath}"`);
                    await siteArchiveImport(env, dataFolderPath); // eslint-disable-line no-await-in-loop
                }

                // restore original site folders after site import
                if (vars.updatedSiteDirectories && vars.updatedSiteDirectories.length) {
                    restoreSiteDirectories(vars.updatedSiteDirectories, { logger });
                }
            } else {
                logger.warn(`${SCRIPT_NAME} There was a problem finding the data folders to upload`);
            }

            // ***********
            // update cartridge paths for the storefront site
            // ***********
            if (socialChannelDetails.storefrontCartridges.length) {
                logger.info(`${SCRIPT_NAME} Adding cartridges to "${vars.siteId}" storefront site cartridge path: ${socialChannelDetails.storefrontCartridges.join(':')}`);
                await updateCartridgePath(socialChannelDetails.storefrontCartridges, vars.siteId, { env, helpers, logger });
            } else {
                logger.warn(`${SCRIPT_NAME} No storefront cartridges to add to the cartridge path`);
            }

            // Find the src where dest is "bm_socialchannels"
            var bmSocialChannelsSrc = socialChannelDetails.cartridges.find(cartridge => cartridge.dest === 'bm_socialchannels').src;
            bmSocialChannelsSrc += '/cartridge/bm_extensions.xml';
            var siteConfigSocialChannel = vars.siteConfigs.RefArch.socialChannel;

            // get current cartridge path via OCAPI
            let siteInfo = await env.ocapi.get(`sites/${vars.siteId}`);
            await modifyXMLAndDownload({ logger }, bmSocialChannelsSrc, siteConfigSocialChannel, siteInfo);
            // ***********
            // update cartridge paths for the business manager site
            // ***********
            if (socialChannelDetails.bizMngrCartridges.length) {
                logger.info(`${SCRIPT_NAME} Adding cartridges to business manager cartridge path: ${socialChannelDetails.bizMngrCartridges.join(':')}`);
                await updateCartridgePath(socialChannelDetails.bizMngrCartridges, BIZ_MNGR_SITE_ID, { env, helpers, logger });
            } else {
                logger.warn(`${SCRIPT_NAME} No business manager cartridges to add to the cartridge path`);
            }

            // ***********
            // update site preferences
            // ***********
            let sitePreferenceUpdateRequests = [];
            let requestBody;
            let sitePreferenceGroup;
            if (vars.socialChannelBase === 'ALL') {
                // get site preference values for all social channels. right now, we only have site prefs for TikTok
                requestBody = getSitePreferenceRequestBody('TIKTOK');
                if (requestBody) {
                    sitePreferenceUpdateRequests.push({
                        requestBody,
                        sitePreferenceGroup: SITE_PREF_GROUPS.TIKTOK
                    });
                }
            } else {
                sitePreferenceGroup = Object.hasOwnProperty.call(SITE_PREF_GROUPS, vars.socialChannelBase)
                    && SITE_PREF_GROUPS[vars.socialChannelBase] ? SITE_PREF_GROUPS[vars.socialChannelBase] : null;
                if (sitePreferenceGroup) {
                    requestBody = getSitePreferenceRequestBody(vars.socialChannelBase);
                    if (requestBody) {
                        sitePreferenceUpdateRequests.push({
                            requestBody,
                            sitePreferenceGroup
                        });
                    }
                }
            }
            if (sitePreferenceUpdateRequests.length) {
                for (let i = 0; i < sitePreferenceUpdateRequests.length; i++) {
                    let sitePreferenceUpdateRequest = sitePreferenceUpdateRequests[i];
                    // eslint-disable-next-line no-await-in-loop
                    await updateSitePreferences({ env, logger }, [vars.siteId], sitePreferenceUpdateRequest.sitePreferenceGroup, sitePreferenceUpdateRequest.requestBody);
                }
            }

            // ***********
            // update OCAPI and WebDAV permissions
            // ***********
            if (vars.ocapiClientId && vars.ocapiClientId !== 'skip') {
                await updateOcapiWebDavPermissions({ env, logger, helpers, vars });
            } else {
                logger.warn(`${SCRIPT_NAME} An OCAPI client ID was not provided. Ensure you follow the documentation to add the correct OCAPI and WebDAV permissions.`);
            }

            // ***********
            // update or create SCAPI client ID
            // ***********
            if (vars.scapiClientId) {
                // skip doing anything with SCAPI clients
                if (vars.scapiClientId === 'skip') {
                    logger.info('Skipping SCAPI client updates');
                    delete vars.scapiClientId;
                    delete vars.scapiClientSecret;
                    return;
                }
                await updateScapiPermissions({ env, logger, helpers, vars });
                await upsertScapiClient({ env, logger, helpers, vars });
            }

            // ***********
            // write and import business manager access role permissions
            // ***********
            await writeAndImportAccessRole({ env, logger, helpers, vars });
        } else {
            logger.info(`${SCRIPT_NAME} data deploy is not enabled, skipping.`);
        }

        // deploy channel specific cartridges to the sandbox
        if (DEPLOY_OPTIONS.CODE_OPTIONS.includes(vars.deployOption)) {
            logger.info(`${SCRIPT_NAME} code deploy is enabled and starting.`);
            if (socialChannelDetails.cartridges.length) {
                if (!env.codeVersion) {
                    try {
                        // set env code version to current on instance
                        let resp = await env.ocapi.get('code_versions');
                        env.codeVersion = resp.data.data.find(c => c.active).id;
                    } catch (e) {
                        throw new Error(`Unable to determine code version: ${e.message}`);
                    }
                }

                let cartridges = socialChannelDetails.cartridges;
                logger.info(`${SCRIPT_NAME} Uploading cartridges: ${cartridges.map(c => c.dest).join(',')} to code version "${env.codeVersion}"`);
                await syncCartridges(env, cartridges, true, { cleanCartridges: true });
            } else {
                logger.warn('There was a problem finding the cartridges to upload');
            }
        } else {
            logger.info(`${SCRIPT_NAME} code deploy is not enabled, skipping.`);
        }
    },
    modifyXMLAndDownload,
    getInstalledSocialChannelsList
};
